<?php

use App\Models\CreditCard;
use App\Models\Transaction;
use App\Support\Currency;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use App\Models\Account;
use App\Models\WalletReserve;
use Livewire\Attributes\Url;

use Livewire\Component;

new class extends Component {
    public $currencySymbol;
    public $cards; // store fetched cards
    public $selectedCard = 'visa';
    public $cardId;
    public $currency = '';
    public $card_level = '';
    public $daily_limit;
    public $accounts; // All user accounts
    public $selectedAccountId; // Currently selected account
    public $accountBalance; // Balance of the selected account
    public $isEditMode = false;
    public $editingCardId = null;
    public $topupAmount = 0;
    public $canModifyCard = false; // Only true for active cards
    public $autoApprovalThreshold = 5000; // Example threshold, can come from config

    public array $currencies = [];

    public array $cardTypes = [];

    public function mount()
    {
        $user = auth()->user();
        $this->currencySymbol = Currency::symbol($user->profile->currency);
        $currencies = collect(json_decode(file_get_contents(resource_path('data/currencies.json')), true));
        $this->currencies = $currencies->sortBy('name')->values()->toArray();

        // Fetch accounts
        $this->accounts = $user->profile->accounts()->get();
        // fetch once
        $this->cards = CreditCard::where('user_id', $user->id)->get();

        // Default selection: first active account
        $activeAccount = $this->accounts->firstWhere('is_active', true) ?? $this->accounts->first();
        if ($activeAccount) {
            $this->selectedAccountId = $activeAccount->id;
            $this->accountBalance = $activeAccount->balance;
        }

        $cardId = request()->query('cardId');

        if ($cardId) {
            $this->editCard($cardId);
        }

        $this->cardTypes = [
            [
                'key' => 'visa',
                'name' => 'Visa',
                'network' => 'VISA',
                'description' => 'Accepted worldwide, suitable for most online purchases',
                'color' => 'blue',
            ],
            [
                'key' => 'mastercard',
                'name' => 'Mastercard',
                'network' => 'MASTERCARD',
                'description' => 'Global acceptance with enhanced security features',
                'color' => 'red',
            ],
            [
                'key' => 'amex',
                'name' => 'American Express',
                'network' => 'AMEX',
                'description' => 'Premium benefits and exclusive rewards program',
                'color' => 'green',
            ],
        ];
    }

    public function updatedSelectedAccountId($value)
    {
        $account = $this->accounts->firstWhere('id', $value);
        if ($account) {
            $this->accountBalance = $account->balance;
        } else {
            $this->accountBalance = 0;
        }
    }

    #[Computed]
    public function activeCards()
    {
        return $this->cards->where('status', 'active')->count();
    }

    #[Computed]
    public function pendingApplications()
    {
        return $this->cards->where('status', 'pending')->count();
    }

    #[Computed]
    public function totalBalance()
    {
        return $this->cards->sum('balance');
    }

    public function openNewCardModal()
    {
        $this->isEditMode = false;
        $this->editingCardId = null;
        $this->canModifyCard = false;

        // Prefill default account for new card
        $activeAccount = $this->accounts->firstWhere('is_active', true) ?? $this->accounts->first();
        if ($activeAccount) {
            $this->selectedAccountId = $activeAccount->id;
            $this->accountBalance = $activeAccount->balance;
        }

        // Reset other fields specific to new card
        $this->selectedCard = 'visa';
        $this->card_level = '';
        $this->currency = '';
        $this->daily_limit = 0;
        $this->topupAmount = 0;
    }

    public function editCard($cardId)
    {
        $card = CreditCard::findOrFail($cardId);

        if (!$card) {
            $this->addError('editingCard', 'Card not found.');
            return;
        }

        $this->isEditMode = true;
        $this->editingCardId = $card->id;

        // Populate form fields
        $this->selectedCard = $card->card_type;
        $this->currency = $card->extra_data['currency'] ?? '';
        $this->card_level = $card->extra_data['card_level'] ?? '';
        $this->daily_limit = $card->extra_data['daily_limit'] ?? 0;
        $this->selectedAccountId = $card->extra_data['linked_account_id'] ?? null;
        $this->accountBalance = optional($this->accounts->firstWhere('id', $this->selectedAccountId))->balance ?? 0;

        // Check if extra actions should be available
        $this->canModifyCard = $card->status === 'active';

        // Reset top-up field
        $this->topupAmount = 0;

        // Open modal via JS event
        $this->dispatch('open-card-modal');
    }

    public function topUpBalance()
    {
        try {
            $this->validate([
                'topupAmount' => ['required', 'numeric', 'min:0.01'],
            ]);

            DB::transaction(function () {
                $scale = 2;

                // Lock the card
                $card = CreditCard::lockForUpdate()->findOrFail($this->editingCardId);

                // Lock linked account
                $accountId = $card->extra_data['linked_account_id'] ?? null;
                $account = Account::lockForUpdate()->findOrFail($accountId);

                // Calculate available balance
                $availableBalance = $account->available_balance; // uses reserves

                if (bccomp((string) $this->topupAmount, $availableBalance, $scale) === 1) {
                    throw new \Exception('Top-up cannot exceed available balance.');
                }

                if (bccomp((string) $this->topupAmount, (string) $this->autoApprovalThreshold, $scale) === 1) {
                    // Pending approval: create a WalletReserve
                    $reserve = WalletReserve::create([
                        'account_id' => $account->id,
                        'action_type' => 'card_topup',
                        'action_id' => $card->id,
                        'amount' => $this->topupAmount,
                        'status' => 'pending',
                        'locked_at' => now(),
                    ]);

                    $this->dispatch('showToast', message: 'Top-up pending admin approval.', type: 'success');
                } else {
                    // Auto top-up: deduct from account, add to card
                    $account->balance = bcsub((string) $account->balance, (string) $this->topupAmount, $scale);
                    $account->save();

                    $card->balance = bcadd((string) $card->balance, (string) $this->topupAmount, $scale);
                    $card->save();

                    $this->dispatch('showToast', message: 'Card topped up successfully.', type: 'success');
                }

                // Update Livewire collections
                $this->cards = $this->cards->map(fn($c) => $c->id == $card->id ? $card : $c);
                $this->accounts = $this->accounts->map(fn($a) => $a->id == $account->id ? $account : $a);

                // Reset input
                $this->topupAmount = 0;
            });
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            $this->addError('topup', 'Card or linked account not found.');
        } catch (\Exception $e) {
            $this->addError('topup', $e->getMessage());
            \Log::error('Top-up error: ' . $e->getMessage());
        }
    }

    public function toggleFreezeCard()
    {
        $card = CreditCard::find($this->editingCardId);

        if (!$card) {
            $this->addError('editingCard', 'Card not found.');
            return;
        }

        // Toggle status
        $card->status = $card->status === 'frozen' ? 'active' : 'frozen';
        $card->save();

        // Refresh collection
        $this->cards = $this->cards->map(fn($c) => $c->id == $card->id ? $card : $c);

        $message = $card->status === 'frozen' ? 'Card frozen.' : 'Card unfrozen.';
        $this->dispatch('showToast', message: $message, type: 'success');
    }

    public function saveCard()
    {
        $this->validate([
            'selectedCard' => 'required|string',
            'currency' => 'required|string|size:3',
            'card_level' => 'required|string',
            'daily_limit' => 'required|numeric|min:0',
        ]);

        $user = auth()->user();

        // Prevent duplicate card type
        $existingCard = CreditCard::where('user_id', $user->id)->where('card_type', $this->selectedCard)->where('id', '!=', $this->editingCardId)->first();

        if ($existingCard) {
            $this->addError('selectedCard', 'You already own a card of this type.');
            return;
        }

        // KYC check
        if (!$user->kyc_verified) {
            $this->addError('daily_limit', 'Complete your KYC before applying for a card.');
            return;
        }

        $account = $this->accounts->firstWhere('id', $this->selectedAccountId);
        if (!$account) {
            $this->addError('selectedAccountId', 'Please select a valid account.');
            return;
        }

        if ($this->daily_limit > $account->available_balance) {
            $this->addError('daily_limit', 'Daily limit cannot exceed the selected account balance.');
            return;
        }

        $minLimits = [
            'standard' => 5,
            'gold' => 10,
            'platinum' => 20,
            'black' => 50,
        ];
        if (isset($minLimits[$this->card_level]) && $this->daily_limit < $minLimits[$this->card_level]) {
            $this->addError('daily_limit', "Minimum daily limit for {$this->card_level} card is {$this->currencySymbol}{$minLimits[$this->card_level]}");
            return;
        }

        try {
            DB::transaction(function () use ($user, $account) {
                if ($this->isEditMode && $this->editingCardId) {
                    // Edit existing card
                    $card = CreditCard::find($this->editingCardId);
                    $card->update([
                        'card_type' => $this->selectedCard,
                        'extra_data' => [
                            'daily_limit' => $this->daily_limit,
                            'card_level' => $this->card_level,
                            'linked_account_id' => $account->id,
                            'currency' => $this->currency,
                        ],
                    ]);

                    // Refresh collection
                    $this->cards = $this->cards->map(fn($c) => $c->id == $card->id ? $card : $c);
                } else {
                    // Create new card
                    $card = CreditCard::create([
                        'user_id' => $user->id,
                        'card_type' => $this->selectedCard,
                        'balance' => 0,
                        'extra_data' => [
                            'daily_limit' => $this->daily_limit,
                            'card_level' => $this->card_level,
                            'linked_account_id' => $account->id,
                            'currency' => $this->currency,
                        ],
                        'status' => 'pending',
                    ]);

                    $this->cards->push($card);
                }
            });

            $this->dispatch('showToast', message: $this->isEditMode ? 'Card updated successfully.' : 'Card application submitted successfully.', type: 'success');
            $this->dispatch('card-success');

            // Reset modal
            $this->resetForm();
        } catch (\Exception $e) {
            \Log::error('Card save failed: ' . $e->getMessage());
            $this->addError('selectedCard', 'Something went wrong. Please try again.');
        }
    }

    public function resetForm()
    {
        $this->reset(['selectedCard', 'card_level', 'currency', 'daily_limit', 'selectedAccountId', 'editingCardId', 'isEditMode']);
        $this->resetValidation();
    }

    public function closeModal()
    {
        $this->resetForm();
    }
};
?>

 <?php $__env->slot('title', null, []); ?> 
    <?php echo e(config('app.name')); ?> || Virtual Cards
 <?php $__env->endSlot(); ?>

<div class="py-3">

    
    <div class="relative block space-y-6">
        <div class="flex justify-end">
            <button type="button" wire:click="openNewCardModal" @click="$dispatch('open-card-modal')"
                class="flex items-center gap-2 bg-linear-to-bl from-sky-500 to-sky-800 p-3 rounded-2xl">
                <?php if (isset($component)) { $__componentOriginal37c717510e7a32140849d8d5dd9d632e = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal37c717510e7a32140849d8d5dd9d632e = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.plus','data' => ['class' => 'w-5 h-5 text-white','variant' => 'solid']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.plus'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-5 h-5 text-white','variant' => 'solid']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal37c717510e7a32140849d8d5dd9d632e)): ?>
<?php $attributes = $__attributesOriginal37c717510e7a32140849d8d5dd9d632e; ?>
<?php unset($__attributesOriginal37c717510e7a32140849d8d5dd9d632e); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal37c717510e7a32140849d8d5dd9d632e)): ?>
<?php $component = $__componentOriginal37c717510e7a32140849d8d5dd9d632e; ?>
<?php unset($__componentOriginal37c717510e7a32140849d8d5dd9d632e); ?>
<?php endif; ?>
                <span class="text-sm capitalize font-semibold text-white">Apply for card</span>
            </button>
        </div>

        <div x-data="{ open: false }" x-show="open" x-cloak x-on:open-card-modal.window="open = true"
            @click.self="open = false; $wire.closeModal()" @keydown.escape.window="open = false; $wire.closeModal()"
            x-on:close-card-modal.window="open = false; $wire.closeModal()"
            x-on:card-success.window="open = false; $wire.closeModal()"
            x-transition:enter="transition ease-out duration-200" x-transition:enter-start="opacity-0 scale-95"
            x-transition:enter-end="opacity-100 scale-100" x-transition:leave="transition ease-in duration-150"
            x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-95"
            x-effect="document.body.classList.toggle('overflow-hidden', open)"
            class="fixed inset-0 z-50 flex items-center justify-center bg-black/30 backdrop-blur-md">

            <!-- Modal content -->
            <div
                class="bg-white dark:bg-gray-800 rounded-xl shadow-lg w-full max-w-2xl
       max-h-[85vh] sm:max-h-[90vh] overflow-y-auto p-6 relative">


                <!-- Close button -->
                <button @click="open = false; $wire.closeModal();"
                    class="sticky top-4 ml-auto block text-gray-500 hover:text-gray-700 dark:hover:text-gray-300">
                    &times;
                </button>


                <div class="space-y-6">

                    
                    <div class="text-center space-y-2">
                        <div
                            class="w-12 h-12 bg-linear-to-bl from-sky-500 to-sky-800 rounded-xl mx-auto flex items-center justify-center text-white">
                            <?php if (isset($component)) { $__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.credit-card','data' => ['variant' => 'solid','class' => 'w-5 h-5']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.credit-card'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-5 h-5']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146)): ?>
<?php $attributes = $__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146; ?>
<?php unset($__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146)): ?>
<?php $component = $__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146; ?>
<?php unset($__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146); ?>
<?php endif; ?>
                        </div>

                        <h2 class="text-base font-semibold tracking-wide capitalize">
                            Apply for a Virtual Card
                        </h2>

                        <span class="text-sm text-gray-600 dark:text-indigo-200">
                            Get instant access to a virtual card for online payments and subscriptions
                        </span>
                    </div>

                    
                    <div class="grid grid-cols-1 gap-4">
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $cardTypes; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $card): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <div class="flex items-center gap-4">
                                <div class="flex items-center justify-center w-5 h-5 border rounded-full border-gray-400"
                                    :class="$wire.selectedCard === '<?php echo e($card['key']); ?>' ?
                                        'bg-<?php echo e($card['color']); ?>-500 border-<?php echo e($card['color']); ?>-500' :
                                        ''">
                                    <div class="w-2 h-2 rounded-full bg-white"
                                        x-show="$wire.selectedCard === '<?php echo e($card['key']); ?>'">
                                    </div>
                                </div>

                                <label @click="$wire.set('selectedCard', '<?php echo e($card['key']); ?>')"
                                    :class="$wire.selectedCard === '<?php echo e($card['key']); ?>' ?
                                        'shadow-lg border-<?php echo e($card['color']); ?>-500 bg-<?php echo e($card['color']); ?>-50 dark:bg-<?php echo e($card['color']); ?>-900' :
                                        'border-gray-300 dark:border-gray-700'"
                                    class="cursor-pointer border rounded-xl p-4 w-full flex flex-col transition-all duration-300 hover:shadow-lg">
                                    <input type="radio" name="card_type" value="<?php echo e($card['key']); ?>"
                                        wire:model="selectedCard" class="sr-only" />

                                    <div class="flex justify-between items-center mb-2">
                                        <span class="font-bold text-gray-900 dark:text-white">
                                            <?php echo e($card['name']); ?>

                                        </span>
                                        <span
                                            class="text-sm font-semibold text-<?php echo e($card['color']); ?>-600 dark:text-<?php echo e($card['color']); ?>-400">
                                            <?php echo e($card['network']); ?>

                                        </span>
                                    </div>

                                    <p class="text-gray-600 dark:text-gray-300 text-sm">
                                        <?php echo e($card['description']); ?>

                                    </p>
                                </label>
                            </div>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php $__errorArgs = ['selectedCard'];
$__bag = $errors->getBag($__errorArgs[1] ?? 'default');
if ($__bag->has($__errorArgs[0])) :
if (isset($message)) { $__messageOriginal = $message; }
$message = $__bag->first($__errorArgs[0]); ?>
                            <p class="mt-1 text-sm text-red-600 dark:text-red-400 flex items-center gap-1">
                                <?php if (isset($component)) { $__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.index','data' => ['name' => 'exclamation-circle','class' => 'h-4 w-4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['name' => 'exclamation-circle','class' => 'h-4 w-4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2)): ?>
<?php $attributes = $__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2; ?>
<?php unset($__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2)): ?>
<?php $component = $__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2; ?>
<?php unset($__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2); ?>
<?php endif; ?>
                                <?php echo e($message); ?>

                            </p>
                        <?php unset($message);
if (isset($__messageOriginal)) { $message = $__messageOriginal; }
endif;
unset($__errorArgs, $__bag); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                    </div>

                    
                    <div>
                        <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model' => 'selectedAccountId','label' => 'Select Account','placeholder' => 'Select Funding Account']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model' => 'selectedAccountId','label' => 'Select Account','placeholder' => 'Select Funding Account']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $accounts; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $account): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                                <option value="<?php echo e($account->id); ?>">
                                    <?php echo e($account->bank_account_type); ?>

                                    - <?php echo e($account->masked_account_number); ?>

                                    - <?php echo e($currencySymbol); ?><?php echo e(number_format($account->available_balance, 2)); ?>

                                </option>
                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                         <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                    </div>

                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                        <div>
                            <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model' => 'card_level','placeholder' => 'select card level','label' => 'Card Level']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model' => 'card_level','placeholder' => 'select card level','label' => 'Card Level']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                <option value="standard">Standard - <?php echo e($authUser->profile->currency); ?>5.00</option>
                                <option value="gold">Gold - <?php echo e($authUser->profile->currency); ?>10.00</option>
                                <option value="platinum">Platinum - <?php echo e($authUser->profile->currency); ?>20.00</option>
                                <option value="black">Black - <?php echo e($authUser->profile->currency); ?>50.00</option>
                             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                        </div>

                        <div>
                            <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model' => 'currency','placeholder' => 'select currency','label' => 'Currency']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model' => 'currency','placeholder' => 'select currency','label' => 'Currency']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $currencies; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $currency): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                                    <option value="<?php echo e($currency['code']); ?>">
                                        <?php echo e($currency['symbol']); ?> <?php echo e($currency['code']); ?>

                                    </option>
                                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                        </div>
                    </div>

                    
                    <div>
                        <?php if (isset($component)) { $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.input','data' => ['wire:model' => 'daily_limit','label' => 'Daily Limit','type' => 'number','step' => '0.01','placeholder' => '$0.00']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model' => 'daily_limit','label' => 'Daily Limit','type' => 'number','step' => '0.01','placeholder' => '$0.00']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $attributes = $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $component = $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
                    </div>

                    <?php echo $__env->make('partials.errors', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>

                    
                    <div x-show="$wire.canModifyCard" class="mt-6 border-t pt-4">
                        <h3 class="font-semibold text-gray-700 dark:text-gray-200 mb-2">
                            Manage Card
                        </h3>

                        
                        <div class="flex gap-2 items-end">
                            <?php if (isset($component)) { $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.input','data' => ['wire:model' => 'topupAmount','label' => 'Top-up Amount','type' => 'number','step' => '0.01']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model' => 'topupAmount','label' => 'Top-up Amount','type' => 'number','step' => '0.01']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $attributes = $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $component = $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
                            <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['wire:click' => 'topUpBalance','variant' => 'secondary']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'topUpBalance','variant' => 'secondary']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                Top-up
                             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
                        </div>

                        
                        <div class="mt-3">
                            <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['wire:click' => 'toggleFreezeCard','variant' => 'danger']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'toggleFreezeCard','variant' => 'danger']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                <?php echo e($cards->firstWhere('id', $editingCardId)?->status === 'frozen' ? 'Unfreeze Card' : 'Freeze Card'); ?>

                             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
                        </div>
                    </div>

                    
                    <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['wire:click' => 'saveCard','variant' => 'primary','class' => 'w-full']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'saveCard','variant' => 'primary','class' => 'w-full']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <?php echo e($isEditMode ? 'Update Card' : 'Submit Application'); ?>

                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>

                    
                    <button type="button" @click="open = false; $wire.closeModal(); "
                        class="w-full text-center text-sm text-red-600 hover:underline">
                        Cancel
                    </button>
                </div>
            </div>
        </div>


        
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <?php
                // Pick first card with a currency symbol, fallback to default
                $symbol = $cards->firstWhere('extra_data.currency')
                    ? Currency::symbol($cards->firstWhere('extra_data.currency')['extra_data']['currency'])
                    : $currencySymbol;

                $total = $cards->sum('balance');
            ?>

            <?php if (isset($component)) { $__componentOriginal88210b161ac9da424ac08c0f17c4d197 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal88210b161ac9da424ac08c0f17c4d197 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.dashboard-card','data' => ['title' => 'TOTAL BALANCE','value' => ''.e($symbol).''.e(number_format($total, 2)).'','icon' => 'credit-card','color' => 'indigo']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('dashboard-card'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['title' => 'TOTAL BALANCE','value' => ''.e($symbol).''.e(number_format($total, 2)).'','icon' => 'credit-card','color' => 'indigo']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal88210b161ac9da424ac08c0f17c4d197)): ?>
<?php $attributes = $__attributesOriginal88210b161ac9da424ac08c0f17c4d197; ?>
<?php unset($__attributesOriginal88210b161ac9da424ac08c0f17c4d197); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal88210b161ac9da424ac08c0f17c4d197)): ?>
<?php $component = $__componentOriginal88210b161ac9da424ac08c0f17c4d197; ?>
<?php unset($__componentOriginal88210b161ac9da424ac08c0f17c4d197); ?>
<?php endif; ?>

            <?php if (isset($component)) { $__componentOriginal88210b161ac9da424ac08c0f17c4d197 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal88210b161ac9da424ac08c0f17c4d197 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.dashboard-card','data' => ['title' => 'ACTIVE CARDS','value' => ''.e($this->activeCards).'','icon' => 'credit-card','color' => 'green']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('dashboard-card'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['title' => 'ACTIVE CARDS','value' => ''.e($this->activeCards).'','icon' => 'credit-card','color' => 'green']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal88210b161ac9da424ac08c0f17c4d197)): ?>
<?php $attributes = $__attributesOriginal88210b161ac9da424ac08c0f17c4d197; ?>
<?php unset($__attributesOriginal88210b161ac9da424ac08c0f17c4d197); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal88210b161ac9da424ac08c0f17c4d197)): ?>
<?php $component = $__componentOriginal88210b161ac9da424ac08c0f17c4d197; ?>
<?php unset($__componentOriginal88210b161ac9da424ac08c0f17c4d197); ?>
<?php endif; ?>
            <?php if (isset($component)) { $__componentOriginal88210b161ac9da424ac08c0f17c4d197 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal88210b161ac9da424ac08c0f17c4d197 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.dashboard-card','data' => ['title' => 'PENDING APPLICATIONS','value' => ''.e($this->pendingApplications).'','icon' => 'clock','color' => 'yellow']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('dashboard-card'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['title' => 'PENDING APPLICATIONS','value' => ''.e($this->pendingApplications).'','icon' => 'clock','color' => 'yellow']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal88210b161ac9da424ac08c0f17c4d197)): ?>
<?php $attributes = $__attributesOriginal88210b161ac9da424ac08c0f17c4d197; ?>
<?php unset($__attributesOriginal88210b161ac9da424ac08c0f17c4d197); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal88210b161ac9da424ac08c0f17c4d197)): ?>
<?php $component = $__componentOriginal88210b161ac9da424ac08c0f17c4d197; ?>
<?php unset($__componentOriginal88210b161ac9da424ac08c0f17c4d197); ?>
<?php endif; ?>
        </div>

        
        <div
            class="bg-linear-to-br from-sky-500 via-sky-600 to-sky-700 dark:from-sky-600 dark:via-sky-700 dark:to-sky-800 rounded-2xl overflow-hidden shadow-xl border border-white/20 dark:border-gray-700/50 mb-4 relative">
            <div class="relative p-6 z-10">
                <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                    <div class="flex-1">
                        <h2 class="text-xl lg:text-2xl font-bold text-white mb-2">Virtual Cards Made Easy</h2>
                        <p class="text-white/90 text-sm mb-4">Create virtual cards for secure online payments,
                            subscription
                            management, and more. Enhanced security and spending control.</p>
                        <div class="grid grid-cols-2 lg:grid-cols-4 gap-3 mb-4">
                            
                            <?php if (isset($component)) { $__componentOriginalfaf842fe2b842383c3be1d62c915509f = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalfaf842fe2b842383c3be1d62c915509f = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.virtual-card-feature','data' => ['icon' => 'shield-check','title' => 'Secure','desc' => 'Protected payments']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('virtual-card-feature'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['icon' => 'shield-check','title' => 'Secure','desc' => 'Protected payments']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $attributes = $__attributesOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $component = $__componentOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__componentOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
                            <?php if (isset($component)) { $__componentOriginalfaf842fe2b842383c3be1d62c915509f = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalfaf842fe2b842383c3be1d62c915509f = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.virtual-card-feature','data' => ['icon' => 'globe-alt','title' => 'Global','desc' => 'Worldwide acceptance']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('virtual-card-feature'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['icon' => 'globe-alt','title' => 'Global','desc' => 'Worldwide acceptance']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $attributes = $__attributesOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $component = $__componentOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__componentOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
                            <?php if (isset($component)) { $__componentOriginalfaf842fe2b842383c3be1d62c915509f = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalfaf842fe2b842383c3be1d62c915509f = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.virtual-card-feature','data' => ['icon' => 'adjustments-horizontal','title' => 'Control','desc' => 'Spending limits']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('virtual-card-feature'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['icon' => 'adjustments-horizontal','title' => 'Control','desc' => 'Spending limits']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $attributes = $__attributesOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $component = $__componentOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__componentOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
                            <?php if (isset($component)) { $__componentOriginalfaf842fe2b842383c3be1d62c915509f = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalfaf842fe2b842383c3be1d62c915509f = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.virtual-card-feature','data' => ['icon' => 'bolt','title' => 'Instant','desc' => 'Quick issuance']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('virtual-card-feature'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['icon' => 'bolt','title' => 'Instant','desc' => 'Quick issuance']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $attributes = $__attributesOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__attributesOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalfaf842fe2b842383c3be1d62c915509f)): ?>
<?php $component = $__componentOriginalfaf842fe2b842383c3be1d62c915509f; ?>
<?php unset($__componentOriginalfaf842fe2b842383c3be1d62c915509f); ?>
<?php endif; ?>
                        </div>
                        <div class="lg:hidden">
                            <a href="#"
                                class="inline-flex items-center px-4 py-2 bg-white/20 backdrop-blur-sm hover:bg-white/30 text-white rounded-xl font-semibold transition-all duration-300 border border-white/30">
                                <i class="fas fa-plus mr-2"></i> Apply Now
                            </a>
                        </div>
                    </div>

                    
                    <div class="hidden lg:flex lg:items-center lg:justify-center lg:w-1/3 lg:pl-6">
                        <?php if (isset($component)) { $__componentOriginal40c11ea655117763d65e37a93ee40ccd = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal40c11ea655117763d65e37a93ee40ccd = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.virtual-card-preview','data' => []] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('virtual-card-preview'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes([]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal40c11ea655117763d65e37a93ee40ccd)): ?>
<?php $attributes = $__attributesOriginal40c11ea655117763d65e37a93ee40ccd; ?>
<?php unset($__attributesOriginal40c11ea655117763d65e37a93ee40ccd); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal40c11ea655117763d65e37a93ee40ccd)): ?>
<?php $component = $__componentOriginal40c11ea655117763d65e37a93ee40ccd; ?>
<?php unset($__componentOriginal40c11ea655117763d65e37a93ee40ccd); ?>
<?php endif; ?>
                    </div>
                </div>
            </div>
            
            <div class="absolute inset-0 opacity-10">
                <div
                    class="absolute top-0 right-0 w-32 h-32 rounded-full bg-white transform translate-x-16 -translate-y-16">
                </div>
                <div
                    class="absolute bottom-0 left-0 w-24 h-24 rounded-full bg-white transform -translate-x-12 translate-y-12">
                </div>
            </div>
        </div>

        
        <div
            class="bg-white/95 dark:bg-gray-800/95 backdrop-blur-xl rounded-2xl shadow-xl border border-white/20 dark:border-gray-700/50 overflow-hidden">
            <div
                class="border-b border-gray-200/50 dark:border-gray-700/50 px-6 py-4 flex justify-between items-center">
                <h2 class="text-lg font-semibold text-gray-900 dark:text-white">Your Cards</h2>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 p-3">
                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__empty_1 = true; $__currentLoopData = $cards; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $card): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); $__empty_1 = false; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                    <?php if (isset($component)) { $__componentOriginalc14740c500b5c5df0af784e37a96759b = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc14740c500b5c5df0af784e37a96759b = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.virtual-card-item','data' => ['card' => $card,'currencySymbol' => $currencySymbol]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('virtual-card-item'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['card' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute($card),'currency-symbol' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute($currencySymbol)]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc14740c500b5c5df0af784e37a96759b)): ?>
<?php $attributes = $__attributesOriginalc14740c500b5c5df0af784e37a96759b; ?>
<?php unset($__attributesOriginalc14740c500b5c5df0af784e37a96759b); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc14740c500b5c5df0af784e37a96759b)): ?>
<?php $component = $__componentOriginalc14740c500b5c5df0af784e37a96759b; ?>
<?php unset($__componentOriginalc14740c500b5c5df0af784e37a96759b); ?>
<?php endif; ?>
                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); if ($__empty_1): ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                    <div class="col-span-full flex justify-center">
                        <div
                            class="flex max-w-md flex-col items-center justify-center gap-4 rounded-xl text-center shadow-sm ">

                            <div
                                class="flex h-14 w-14 items-center justify-center rounded-full bg-sky-100 dark:bg-sky-500/20">
                                <?php if (isset($component)) { $__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.index','data' => ['name' => 'credit-card','class' => 'h-6 w-6 text-white','variant' => 'solid']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['name' => 'credit-card','class' => 'h-6 w-6 text-white','variant' => 'solid']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2)): ?>
<?php $attributes = $__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2; ?>
<?php unset($__attributesOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2)): ?>
<?php $component = $__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2; ?>
<?php unset($__componentOriginalc7d5f44bf2a2d803ed0b55f72f1f82e2); ?>
<?php endif; ?>
                            </div>

                            <h3 class="text-lg font-semibold text-gray-800 dark:text-sky-100 capitalize">
                                No cards yet
                            </h3>

                            <p class="text-sm text-gray-600 dark:text-sky-300">
                                You haven’t created a virtual card yet. Apply now and start making secure online
                                payments in minutes.
                            </p>

                            <button type="button" wire:click="openNewCardModal"
                                @click="$dispatch('open-card-modal')"
                                class="flex items-center gap-2 bg-linear-to-bl from-sky-500 to-sky-800 p-3 rounded-2xl">
                                <?php if (isset($component)) { $__componentOriginal37c717510e7a32140849d8d5dd9d632e = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal37c717510e7a32140849d8d5dd9d632e = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.plus','data' => ['class' => 'w-5 h-5 text-white','variant' => 'solid']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.plus'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-5 h-5 text-white','variant' => 'solid']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal37c717510e7a32140849d8d5dd9d632e)): ?>
<?php $attributes = $__attributesOriginal37c717510e7a32140849d8d5dd9d632e; ?>
<?php unset($__attributesOriginal37c717510e7a32140849d8d5dd9d632e); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal37c717510e7a32140849d8d5dd9d632e)): ?>
<?php $component = $__componentOriginal37c717510e7a32140849d8d5dd9d632e; ?>
<?php unset($__componentOriginal37c717510e7a32140849d8d5dd9d632e); ?>
<?php endif; ?>
                                <span class="text-sm capitalize font-semibold text-white">Apply for a new card</span>
                            </button>
                        </div>
                    </div>
                <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
            </div>

        </div>

    </div>

</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/pages/user/⚡cards.blade.php ENDPATH**/ ?>