<?php

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\User;
use App\Models\Kyc;
use Illuminate\Support\Facades\Storage;
use App\Notifications\KycStatusNotification;

new class extends Component {
    use WithFileUploads;

    public User $user;
    public $photo; // For wire:model upload

    public function mount(User $user)
    {
        $this->user = $user->load('kyc'); // load all KYC documents
    }

    // 1️⃣ Define rules
    protected $rules = [
        'photo' => 'required|image|max:10240|mimes:jpg,jpeg,png,webp',
    ];

    // 2️⃣ Define custom messages
    protected $messages = [
        'photo.required' => 'Please upload a photo.',
        'photo.image' => 'The uploaded file must be an image.type: jpg, jpeg, png, webp',
        'photo.max' => 'The photo must not be larger than 10 MB.',
        'photo.mimes' => 'The photo must be a file of type: jpg, jpeg, png, webp.',
    ];

    /** PROFILE PHOTO */
    public function updateProfilePhoto()
    {
        $this->validate();
        $profile = $this->user->profile ?? $this->user->profile()->create([]);

        // Delete old photo if exists
        if ($profile->profile_photo && Storage::disk('public')->exists($profile->profile_photo)) {
            Storage::disk('public')->delete($profile->profile_photo);
        }

        // Store new photo
        $path = $this->photo->store('avatars', 'public');

        $profile->update([
            'profile_photo' => $path,
        ]);

        $publicUrl = Storage::disk('public')->url($path) . '?t=' . time();

        $this->dispatch('profile-image-updated', name: $this->user->name, photoUrl: $publicUrl . '?t=' . time());

        $this->dispatch('showToast', message: 'Profile image updated successfully.', type: 'success');

        $this->reset('photo');
    }

    /** OPEN REJECT MODAL */
    public function openRejectModal(int $documentId): void
    {
        $this->dispatch('open-reject-modal', id: $documentId);
    }

    /** CONFIRM REJECTION (called from AlpineJS) */
    public function confirmReject(int $documentId, string $reason): void
    {
        $this->rejectKycDocument($documentId, $reason);
    }

    /** APPROVE KYC DOCUMENT */
    public function approveKycDocument(int $id): void
    {
        $document = Kyc::findOrFail($id);

        if ($document->status === Kyc::STATUS_VERIFIED) {
            return;
        }

        $document->update([
            'status' => Kyc::STATUS_VERIFIED,
            'verified_at' => now(),
            'rejection_reason' => null,
        ]);
        $documentType = $document->id_type ?? 'Unknown';
        $document->user->notify(new KycStatusNotification(documentType: $documentType, status: 'approved'));
        $this->dispatch('notification-sent');
        $this->dispatch('showToast', message: 'Document approved and user notified.', type: 'success');
    }

    /** REJECT KYC DOCUMENT */
    public function rejectKycDocument(int $documentId, string $reason): void
    {
        $reason = trim($reason);
        if (strlen($reason) < 5) {
            $this->addError('reason', 'Rejection reason must be at least 5 characters');
            return;
        }

        $document = Kyc::findOrFail($documentId);

        if ($document->status === Kyc::STATUS_REJECTED) {
            return;
        }

        $document->update([
            'status' => Kyc::STATUS_REJECTED,
            'rejection_reason' => $reason,
            'verified_at' => null,
        ]);

        $documentType = $document->id_type ?? 'Unknown';
        $document->user->notify(new KycStatusNotification(documentType: $documentType, status: 'rejected', reason: $reason));
        $this->dispatch('notification-sent');
        $this->dispatch('reject-success');
        $this->dispatch('showToast', message: 'Document rejected and user notified.', type: 'success');
    }
};
?>
<div>
    <div class="lg:flex lg:space-x-6">

        
        <div
            class="lg:w-1/3 bg-linear-to-b from-gray-50 to-white dark:from-gray-900 dark:to-gray-800 p-6 rounded-xl shadow-lg flex flex-col items-center space-y-6">

            
            
            <div x-data="{
                photoPreview: null,
                showSaveButton: false,
                description: 'Choose a photo to update your profile',
                originalDescription: 'Choose a photo to update your profile',
                uploadProgress: 0,
                uploading: false,
            
                init() {
                    // Start upload
                    this.$el.addEventListener('livewire-upload-start', () => {
                        this.uploading = true;
                        this.uploadProgress = 0;
                        this.description = 'Uploading... 0%';
                        this.showSaveButton = false;
                    });
            
                    // Progress
                    this.$el.addEventListener('livewire-upload-progress', event => {
                        this.uploadProgress = event.detail.progress;
                        this.description = 'Uploading... ' + this.uploadProgress + '%';
                    });
            
                    // Upload finished
                    this.$el.addEventListener('livewire-upload-finish', () => {
                        this.uploading = false;
                        this.description = this.originalDescription;
                        this.showSaveButton = true;
                    });
            
                    // Upload error
                    this.$el.addEventListener('livewire-upload-error', () => {
                        this.uploading = false;
                        this.description = 'Upload failed. Try again.';
                        this.showSaveButton = false;
                    });
            
                    // Profile updated
                    window.addEventListener('profile-image-updated', e => {
                        this.photoPreview = e.detail.photoUrl;
                        this.showSaveButton = false;
                        this.description = this.originalDescription;
                    });
                },
            
                handleFileChange(event) {
                    const file = event.target.files[0];
                    if (file) {
                        const reader = new FileReader();
                        reader.onload = e => this.photoPreview = e.target.result;
                        reader.readAsDataURL(file);
            
                        // Do NOT show Save button yet — will show after upload finish
                        this.showSaveButton = false;
                    } else {
                        this.photoPreview = null;
                        this.showSaveButton = false;
                    }
                }
            }" class="flex flex-col items-center space-y-2">

                <div class="relative group cursor-pointer" @click="$refs.photoInput.click()">
                    <div class="h-28 w-28 rounded-full overflow-hidden border-4 border-sky-500 shadow relative">
                        <template x-if="photoPreview">
                            <img :src="photoPreview" class="h-full w-full object-cover" />
                        </template>
                        <template x-if="!photoPreview">
                            <img src="<?php echo e(optional($user->profile)->profile_photo
                                ? Storage::disk('public')->url($user->profile->profile_photo)
                                : 'https://ui-avatars.com/api/?name=' . urlencode($user->name)); ?>"
                                alt="<?php echo e($user->name); ?>’s profile photo"
                                class="h-full w-full object-cover transition-opacity duration-300" alt="Profile Photo">
                        </template>

                        <div x-show="uploading"
                            class="absolute inset-0 bg-black/50 flex flex-col items-center justify-center text-white text-xs font-medium transition">
                            Uploading... <span x-text="uploadProgress + '%'"></span>
                        </div>
                    </div>

                    <div
                        class="absolute inset-0 bg-black/50 flex items-center justify-center text-white text-xs font-medium opacity-0 group-hover:opacity-100 transition rounded-full">
                        Change Photo
                    </div>
                </div>
                <p class="text-xs text-gray-500" x-text="description"></p>

                <!-- File Input -->
                <input type="file" x-ref="photoInput" wire:model="photo" class="hidden"
                    @change="handleFileChange($event)" />

                <!-- Display validation error -->
                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php $__errorArgs = ['photo'];
$__bag = $errors->getBag($__errorArgs[1] ?? 'default');
if ($__bag->has($__errorArgs[0])) :
if (isset($message)) { $__messageOriginal = $message; }
$message = $__bag->first($__errorArgs[0]); ?>
                    <p class="text-xs text-red-500 mt-1"><?php echo e($message); ?></p>
                <?php unset($message);
if (isset($__messageOriginal)) { $message = $__messageOriginal; }
endif;
unset($__errorArgs, $__bag); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>

                <!-- Save Button -->
                <button x-show="showSaveButton" x-cloak x-transition wire:click="updateProfilePhoto"
                    wire:loading.attr="disabled" class="text-xs text-sky-600 hover:underline flex items-center gap-2"
                    :disabled="uploading">
                    <span wire:loading.remove>Save Photo</span>
                    <span wire:loading>
                        <svg class="animate-spin h-4 w-4 text-sky-600" xmlns="http://www.w3.org/2000/svg" fill="none"
                            viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor"
                                stroke-width="4">
                            </circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v8H4z"></path>
                        </svg>
                        Saving...
                    </span>
                </button>
            </div>

            
            <div class="text-center space-y-1">
                <h2 class="text-xl font-semibold text-gray-800 dark:text-gray-100">
                    <?php echo e($user->name); ?>

                </h2>
                <p class="text-sm text-gray-500 dark:text-gray-400">
                    <?php echo e($user->email); ?>

                </p>
            </div>

            
            <div class="w-full space-y-3 text-sm">
                <div class="flex justify-between">
                    <span class="text-gray-600 dark:text-gray-300">Account Created</span>
                    <span class="font-medium text-gray-800 dark:text-gray-100">
                        <?php echo e($user->created_at->format('M d, Y')); ?>

                    </span>
                </div>

                <?php
                    $latestKyc = $user->kyc->firstWhere('is_latest', true); // LEFT column
                    $sortedKyc = $user->kyc->sortByDesc('is_latest'); // RIGHT column
                ?>

                <div class="flex justify-between items-center">
                    <span class="text-gray-600 dark:text-gray-300">KYC Status</span>

                    <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($latestKyc): ?>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($latestKyc->isRejected()): ?>
                            <span class="bg-red-100 text-red-800 px-2 py-1 rounded-full text-xs font-semibold">
                                Rejected
                            </span>
                        <?php elseif($latestKyc->isVerified()): ?>
                            <span class="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs font-semibold">
                                Verified
                            </span>
                        <?php else: ?>
                            <span class="bg-yellow-100 text-yellow-800 px-2 py-1 rounded-full text-xs font-semibold">
                                Pending Review
                            </span>
                        <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                    <?php else: ?>
                        <span class="bg-gray-100 text-gray-800 px-2 py-1 rounded-full text-xs font-semibold">
                            No KYC Uploaded
                        </span>
                    <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                </div>
            </div>
        </div>

        
        <div class="lg:w-2/3 bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg">

            <h2 class="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-6">
                KYC Documents
            </h2>

            <div class="space-y-4">

                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__empty_1 = true; $__currentLoopData = $sortedKyc; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $document): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); $__empty_1 = false; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                    <div
                        class="border dark:border-gray-700 rounded-lg p-4 space-y-4
                        <?php echo e($document->is_latest ? 'ring-2 ring-sky-500 dark:ring-sky-400' : ''); ?>">

                        
                        <div class="flex justify-between items-center">
                            <h3 class="font-medium text-gray-800 dark:text-gray-100">
                                <?php echo e(ucfirst($document->id_type)); ?>

                                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($document->is_latest): ?>
                                    <span class="ml-2 text-xs text-sky-600 dark:text-sky-400">(Latest)</span>
                                <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                            </h3>

                            
                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($document->isVerified()): ?>
                                <span class="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs font-semibold">
                                    Verified
                                </span>
                            <?php elseif($document->isRejected()): ?>
                                <span class="bg-red-100 text-red-800 px-2 py-1 rounded-full text-xs font-semibold">
                                    Rejected
                                </span>
                            <?php else: ?>
                                <span
                                    class="bg-yellow-100 text-yellow-800 px-2 py-1 rounded-full text-xs font-semibold">
                                    Awaiting Review
                                </span>
                            <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                        </div>

                        
                        <div class="space-y-1">
                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $document->id_document_paths ?? []; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $path): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if(Storage::disk('public')->exists($path)): ?>
                                    <a href="<?php echo e(Storage::disk('public')->url($path)); ?>" target="_blank"
                                        class="text-sky-600 hover:underline text-sm block">
                                        View Uploaded Document
                                    </a>
                                <?php else: ?>
                                    <span class="text-red-500 text-xs block">
                                        File Missing
                                    </span>
                                <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                        </div>
                        
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($document->isPending()): ?>
                            <div class="flex space-x-3">
                                <?php if (isset($component)) { $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::button.index','data' => ['wire:click' => 'approveKycDocument('.e($document->id).')','type' => 'button','variant' => 'primary','color' => 'green','class' => 'flex-1 text-xs']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'approveKycDocument('.e($document->id).')','type' => 'button','variant' => 'primary','color' => 'green','class' => 'flex-1 text-xs']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                    Approve Document
                                 <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $attributes = $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $component = $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>

                                <?php if (isset($component)) { $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::button.index','data' => ['type' => 'button','variant' => 'danger','wire:click' => 'openRejectModal('.e($document->id).')','class' => 'flex-1 text-xs']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['type' => 'button','variant' => 'danger','wire:click' => 'openRejectModal('.e($document->id).')','class' => 'flex-1 text-xs']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                    Reject Document
                                 <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $attributes = $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $component = $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
                            </div>
                        <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                    </div>
                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); if ($__empty_1): ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                    <div class="text-center text-gray-500 dark:text-gray-400 text-sm py-6">
                        No KYC documents uploaded.
                    </div>
                <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>

            </div>
        </div>
    </div>

    
    <div x-data="{
        open: false,
        documentId: null,
        reason: '',
        closeModal() {
            this.open = false;
            this.documentId = null;
            this.reason = '';
        }
    }"
        x-on:open-reject-modal.window="
        open = true;
        documentId = $event.detail.id;
        reason = '';
    "
        x-on:reject-success.window="closeModal()" x-show="open" x-cloak
        class="fixed inset-0 z-50 flex items-center justify-center bg-black/50">
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-xl w-full max-w-md p-6 space-y-4"
            @click.away="closeModal()">

            <h2 class="text-lg font-semibold text-gray-800 dark:text-gray-100">
                Reject KYC Document
            </h2>

            <p class="text-sm text-gray-600 dark:text-gray-400">
                Please provide a reason for rejection.
                This explanation will be visible to the user.
            </p>

            <div>
                <?php if (isset($component)) { $__componentOriginal4727f9fd7c3055c2cf9c658d89b16886 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.textarea','data' => ['wire:model.defer' => 'reason','label' => 'Rejection Reason','xModel' => 'reason','rows' => '4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('textarea'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'reason','label' => 'Rejection Reason','x-model' => 'reason','rows' => '4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886)): ?>
<?php $attributes = $__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886; ?>
<?php unset($__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal4727f9fd7c3055c2cf9c658d89b16886)): ?>
<?php $component = $__componentOriginal4727f9fd7c3055c2cf9c658d89b16886; ?>
<?php unset($__componentOriginal4727f9fd7c3055c2cf9c658d89b16886); ?>
<?php endif; ?>
            </div>

            <div class="flex justify-end space-x-3 pt-2">
                <button type="button" @click="closeModal()"
                    class="px-4 py-2 text-sm bg-gray-200 dark:bg-gray-600 rounded-md">
                    Cancel
                </button>

                <?php if (isset($component)) { $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::button.index','data' => ['type' => 'button','wire:loading.attr' => 'disabled','wire:target' => 'confirmReject','@click' => '$wire.call(\'confirmReject\', documentId, reason)','icon' => 'exclamation-circle','variant' => 'danger']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['type' => 'button','wire:loading.attr' => 'disabled','wire:target' => 'confirmReject','@click' => '$wire.call(\'confirmReject\', documentId, reason)','icon' => 'exclamation-circle','variant' => 'danger']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                    <span wire:loading.remove wire:target="confirmReject">Reject</span>
                    <span wire:loading wire:target="confirmReject">Rejecting...</span>
                 <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $attributes = $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $component = $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>

            </div>
        </div>
    </div>

</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/components/user/show/⚡user-kyc.blade.php ENDPATH**/ ?>