<?php

use Livewire\Component;
use App\Models\Account;
use App\Models\Beneficiary;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Collection;

new class extends Component {
    public ?int $selectedBeneficiary = null;
    public Collection $beneficiaries;
    public $allInternationalMethods;
    public ?int $selectedPaymentMethod = null;
    public string $transferType = Beneficiary::LOCAL; // default
    public ?string $selectedPaymentMethodType = null;

    public array $newBeneficiary = [
        'account_number' => '',
        'nickname' => null,
        'recipient_name' => null,
        'beneficiary_name' => null,
        'bank_name' => null,
        'bank_address' => null,
        'country' => null,
        'swift_code' => null,
        'iban' => null,
        'routing_number' => null,
        'currency' => null,
        'coin' => null,
        'network' => null,
        'wallet_address' => null,
        'paypal_email' => null,
        'transaction_hash' => null,
        'reference' => null,
        'note' => null,
    ];

    // -----------------------------
    // Component Lifecycle
    // -----------------------------
    public function mount(string $transferType = Beneficiary::LOCAL): void
    {
        $this->transferType = $transferType;
        $this->beneficiaries = collect();

        // Get all active international methods
        $this->allInternationalMethods = \App\Models\PaymentMethod::where('type', 'international_transfer')->where('is_active', true)->get();

        // Preselect first method for international transfers
        if ($transferType === Beneficiary::INTERNATIONAL && $this->allInternationalMethods->isNotEmpty()) {
            $this->selectedPaymentMethod = $this->allInternationalMethods->first()->id;
            $this->updatedSelectedPaymentMethod();
        }

        $this->refreshBeneficiaries();
    }

    // -----------------------------
    // Computed Properties
    // -----------------------------
    #[Computed]
    public function getCurrentPaymentFieldsProperty(): array
    {
        $method = \App\Models\PaymentMethod::find($this->selectedPaymentMethod);
        return $method->fields['fields'] ?? [];
    }

    // -----------------------------
    // Event Handlers
    // -----------------------------
    public function updatedSelectedPaymentMethod(): void
    {
        $method = \App\Models\PaymentMethod::find($this->selectedPaymentMethod);

        if (!$method) {
            $this->selectedPaymentMethodType = null;
            return;
        }

        $nameLower = strtolower($method->name);

        // Override type for crypto/paypal based on name
        if (str_contains($nameLower, 'crypto')) {
            $this->selectedPaymentMethodType = 'crypto';
        } elseif (str_contains($nameLower, 'paypal')) {
            $this->selectedPaymentMethodType = 'paypal';
        } else {
            $this->selectedPaymentMethodType = $method->type; // e.g., international_transfer, local
        }

        // Clear irrelevant fields for crypto/paypal
        if (in_array($this->selectedPaymentMethodType, ['crypto', 'paypal'])) {
            foreach (['account_number', 'bank_name', 'bank_address', 'swift_code', 'iban', 'routing_number'] as $field) {
                $this->newBeneficiary[$field] = null;
            }
        }
    }

    #[On('beneficiary-added')]
    public function refreshBeneficiaries($id = null): void
    {
        $user = auth()->user();
        $query = $user->beneficiaries();

        $query = $this->transferType === Beneficiary::INTERNATIONAL ? $query->international() : $query->local();

        $this->beneficiaries = $query->get();

        if ($id) {
            $this->selectedBeneficiary = $id;
        }
    }

    // -----------------------------
    // Beneficiary Selection
    // -----------------------------
    public function selectBeneficiary(int $id): void
    {
        $this->selectedBeneficiary = $id;

        $beneficiary = Beneficiary::find($id);
        if (!$beneficiary) {
            return;
        }

        $methods = \App\Models\PaymentMethod::where('type', 'international_transfer')->where('is_active', true)->get();

        $paymentMethod = $methods->first(fn($m) => $m->id === ($beneficiary->meta['payment_method_id'] ?? null)) ?? $methods->first();

        $this->selectedPaymentMethod = $paymentMethod->id ?? null;
        $this->updatedSelectedPaymentMethod();

        // Autofill newBeneficiary fields
        foreach ($beneficiary->meta ?? [] as $key => $value) {
            $this->newBeneficiary[$key] = $value;
        }

        $this->newBeneficiary['account_number'] = $beneficiary->account_number;
        $this->newBeneficiary['nickname'] = $beneficiary->nickname;

        $this->dispatch('beneficiary-selected', id: $id, paymentMethodName: $paymentMethod->name ?? null);
    }

    // -----------------------------
    // Beneficiary Management
    // -----------------------------
    public function addBeneficiary(): void
    {
        $this->validate($this->getValidationRules());

        $user = auth()->user();
        $acctNum = $this->newBeneficiary['account_number'];

        // Prevent adding user's own account
        $userAccountNumbers = $user->profile?->accounts()->pluck('account_number')->toArray() ?? [];
        if (in_array($acctNum, $userAccountNumbers)) {
            throw ValidationException::withMessages([
                'newBeneficiary.account_number' => 'You cannot add your own account as a beneficiary.',
            ]);
        }

        // Validate local account exists
        if ($this->transferType === Beneficiary::LOCAL) {
            $account = $this->checkAccountNumber($acctNum);
            if (!$account?->profile?->user) {
                throw ValidationException::withMessages([
                    'newBeneficiary.account_number' => 'Account number not found.',
                ]);
            }
        }

        // Build meta
        $meta = collect($this->newBeneficiary)
            ->except(['account_number', 'nickname'])
            ->filter(fn($v) => $v !== null)
            ->toArray();

        $beneficiary = Beneficiary::create([
            'user_id' => $user->id,
            'nickname' => $this->newBeneficiary['nickname'] ?? null,
            'account_number' => $acctNum,
            'type' => $this->transferType,
            'meta' => array_merge(['payment_method_id' => $this->selectedPaymentMethod], $meta),
        ]);

        $this->beneficiaries->push($beneficiary);
        $this->reset(['newBeneficiary']);
        $this->resetErrorBag();

        $this->dispatch('beneficiary-added', id: $beneficiary->id);
        $this->dispatch('showToast', message: 'Beneficiary added successfully.', type: 'success');
    }

    // -----------------------------
    // Validation
    // -----------------------------
    protected function getValidationRules(): array
    {
        $rules = ['newBeneficiary.nickname' => ['nullable', 'string', 'max:255']];
        $excludedTypes = ['crypto', 'paypal'];

        if ($this->transferType === Beneficiary::LOCAL || !in_array($this->selectedPaymentMethodType, $excludedTypes)) {
            $rules['newBeneficiary.account_number'] = ['required', 'numeric', Rule::unique('beneficiaries', 'account_number')->where('user_id', auth()->id())];
        }

        foreach ($this->currentPaymentFields as $field) {
            $key = 'newBeneficiary.' . $field['name'];
            if (empty($field['required'])) {
                $rules[$key] = 'nullable';
                continue;
            }

            $rules[$key] = match ($field['type'] ?? 'text') {
                'text', 'textarea' => 'required|string|max:255',
                'number' => 'required|numeric|min:1',
                'select' => ['required', Rule::in(array_map(fn($o) => $o['value'] ?? $o, $field['options'] ?? []))],
                default => 'required|string|max:255',
            };
        }

        return $rules;
    }

    protected function getValidationMessages(): array
    {
        $messages = [];
        foreach ($this->currentPaymentFields as $field) {
            $key = 'newBeneficiary.' . $field['name'];
            $label = $field['label'] ?? ucfirst(str_replace('_', ' ', $field['name']));

            if ($field['type'] === 'file') {
                $messages["$key.required"] = "$label is required.";
                $messages["$key.file"] = "$label must be a valid file.";
                $messages["$key.mimes"] = "$label must be jpg, png or pdf.";
                $messages["$key.max"] = "$label may not exceed 2MB.";
            } elseif ($field['type'] === 'number') {
                $messages["$key.required"] = "$label is required.";
                $messages["$key.numeric"] = "$label must be numeric.";
                $messages["$key.min"] = "$label must be at least 1.";
            } else {
                $messages["$key.required"] = "$label is required.";
                $messages["$key.string"] = "$label must be a valid string.";
                $messages["$key.max"] = "$label may not exceed 255 characters.";
            }
        }

        // Static messages
        $messages['newBeneficiary.account_number.required'] = 'Account number is required.';
        $messages['newBeneficiary.account_number.numeric'] = 'Account number must be numeric.';
        $messages['newBeneficiary.account_number.unique'] = 'You already added this account as a beneficiary.';

        return $messages;
    }

    private function checkAccountNumber(string $accountNumber): ?Account
    {
        return Account::where('account_number', $accountNumber)->with('profile.user')->first();
    }

    public function closeModal(): void
    {
        $this->reset(['newBeneficiary']);
        $this->resetErrorBag();
    }
};
?>
<div class="rounded-xl p-4 shadow-sm dark:shadow-gray-900/25 border border-gray-100 dark:border-gray-700">

    <!-- Heading -->
    <div class="flex items-center justify-between text-gray-800 dark:text-gray-200 mb-4">
        <h3 class="text-base font-semibold text-gray-900 dark:text-white"><?php echo e($title ?? 'Beneficiaries'); ?></h3>
    </div>

    <!-- Beneficiaries Flex Section -->
    <div x-data="{ showAll: false }" class="flex space-x-3 overflow-x-auto pb-2">

        <!-- Add Beneficiary Button -->
        <button type="button" x-data @click="$dispatch('open-modal')"
            class="shrink-0 flex flex-col items-center justify-center group">
            <div
                class="w-16 h-16 rounded-full border-2 border-dashed border-gray-300 dark:border-gray-600 bg-gray-50 dark:bg-gray-800 flex items-center justify-center mb-2 group-hover:bg-gray-100 dark:group-hover:bg-gray-700 group-hover:border-primary-400 dark:group-hover:border-primary-500 transition-all duration-200">
                <?php if (isset($component)) { $__componentOriginal37c717510e7a32140849d8d5dd9d632e = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal37c717510e7a32140849d8d5dd9d632e = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.plus','data' => ['variant' => 'solid','class' => 'w-5 h-5']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.plus'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-5 h-5']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal37c717510e7a32140849d8d5dd9d632e)): ?>
<?php $attributes = $__attributesOriginal37c717510e7a32140849d8d5dd9d632e; ?>
<?php unset($__attributesOriginal37c717510e7a32140849d8d5dd9d632e); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal37c717510e7a32140849d8d5dd9d632e)): ?>
<?php $component = $__componentOriginal37c717510e7a32140849d8d5dd9d632e; ?>
<?php unset($__componentOriginal37c717510e7a32140849d8d5dd9d632e); ?>
<?php endif; ?>
            </div>
            <span
                class="text-xs text-gray-500 dark:text-gray-400 text-center group-hover:text-primary-600 dark:group-hover:text-primary-400 transition-colors">
                Add New
            </span>
        </button>

        <!-- Existing Beneficiaries -->
        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__empty_1 = true; $__currentLoopData = $beneficiaries; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $index => $beneficiary): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); $__empty_1 = false; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
            <div x-show="showAll || <?php echo e($index); ?> < 6" x-transition:enter="transition ease-out duration-300"
                x-transition:enter-start="opacity-0 translate-y-2" x-transition:enter-end="opacity-100 translate-y-0"
                x-transition:leave="transition ease-in duration-200"
                x-transition:leave-start="opacity-100 translate-y-0" x-transition:leave-end="opacity-0 translate-y-2"
                class="shrink-0 flex flex-col items-center justify-center group relative">

                <button wire:click="selectBeneficiary(<?php echo e($beneficiary->id); ?>)"
                    class="relative w-16 h-16 rounded-full flex items-center justify-center mb-2 border-2 border-white dark:border-gray-800 shadow-sm dark:shadow-gray-900/25 hover:scale-105 transition-all duration-200 <?php echo e($beneficiary->color); ?> <?php echo e($selectedBeneficiary === $beneficiary->id ? 'ring-4 ring-sky-500' : ''); ?>"
                    title="<?php echo e(($beneficiary->meta['recipient_name'] ?? $beneficiary->nickname) . ' - Acct: ' . $beneficiary->account_number); ?>">
                    <?php echo e($beneficiary->initials); ?>

                    <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($selectedBeneficiary === $beneficiary->id): ?>
                        <span
                            class="absolute bottom-0 -right-1 w-5 h-5 bg-green-500 rounded-full flex items-center justify-center text-white text-xs shadow-lg">
                            ✓
                        </span>
                    <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                </button>
            </div>
        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); if ($__empty_1): ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
            <div class="shrink-0 flex flex-col items-center justify-center ">
                <div class="w-16 h-16 rounded-full bg-gray-100 dark:bg-gray-700 flex items-center justify-center mb-3">
                    <?php if (isset($component)) { $__componentOriginaldeeed84ca4f29f425b89c454233ef87a = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaldeeed84ca4f29f425b89c454233ef87a = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.user-group','data' => ['variant' => 'solid','class' => 'w-5 h-5']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.user-group'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-5 h-5']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaldeeed84ca4f29f425b89c454233ef87a)): ?>
<?php $attributes = $__attributesOriginaldeeed84ca4f29f425b89c454233ef87a; ?>
<?php unset($__attributesOriginaldeeed84ca4f29f425b89c454233ef87a); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaldeeed84ca4f29f425b89c454233ef87a)): ?>
<?php $component = $__componentOriginaldeeed84ca4f29f425b89c454233ef87a; ?>
<?php unset($__componentOriginaldeeed84ca4f29f425b89c454233ef87a); ?>
<?php endif; ?>
                </div>
                <p class="text-sm text-gray-500 dark:text-gray-400 text-center">No saved beneficiaries</p>
                <p class="text-xs text-gray-400 dark:text-gray-500 text-center mt-1">Add one to get started
                </p>
            </div>
        <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>

        <!-- Show More Button -->
        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($beneficiaries->count() > 6): ?>
            <button @click="showAll = !showAll"
                class="flex items-center justify-center w-14 h-14 rounded-full border border-gray-300 dark:border-gray-600 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:ring hover:ring-sky-200 dark:hover:ring-sky-700 transition">
                <span x-text="showAll ? '-' : '+'"></span>
            </button>
        <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>

    </div>

    <!-- Add Beneficiary Modal -->
    <div x-data="{ open: false }" x-cloak x-show="open" x-on:open-modal.window="open = true"
        x-on:beneficiary-added.window="open = false" x-on:keydown.escape.window="open = false; $wire.closeModal()"
        class="fixed inset-0 z-50 flex items-center justify-center bg-black/60">
        <div @click.away="open = false; $wire.closeModal()"
            class="bg-white dark:bg-gray-800 rounded-xl shadow-lg w-full max-w-3xl p-6 max-h-[90vh] overflow-auto">

            <h2 class="text-lg font-semibold mb-4 text-gray-900 dark:text-white">
                Add Beneficiary
            </h2>

            <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 pb-4">
                <!-- Local/common fields -->
                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($selectedPaymentMethodType !== 'crypto' && $selectedPaymentMethodType !== 'paypal'): ?>
                    <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'newBeneficiary.account_number','label' => 'Account Number','type' => 'number']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'newBeneficiary.account_number','label' => 'Account Number','type' => 'number']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
                <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>

                <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'newBeneficiary.nickname','label' => 'Nickname','type' => 'text']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'newBeneficiary.nickname','label' => 'Nickname','type' => 'text']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>

                <!-- International fields -->
                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($transferType === Beneficiary::INTERNATIONAL): ?>

                    <?php if (isset($component)) { $__componentOriginala467913f9ff34913553be64599ec6e92 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginala467913f9ff34913553be64599ec6e92 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::select.index','data' => ['wire:model.live.debounce.500ms' => 'selectedPaymentMethod','label' => 'Transfer Method']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.live.debounce.500ms' => 'selectedPaymentMethod','label' => 'Transfer Method']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <option value="" disabled>Select Method</option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $allInternationalMethods; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $method): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <option value="<?php echo e($method->id); ?>"><?php echo e($method->name); ?></option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginala467913f9ff34913553be64599ec6e92)): ?>
<?php $attributes = $__attributesOriginala467913f9ff34913553be64599ec6e92; ?>
<?php unset($__attributesOriginala467913f9ff34913553be64599ec6e92); ?>
<?php endif; ?>
<?php if (isset($__componentOriginala467913f9ff34913553be64599ec6e92)): ?>
<?php $component = $__componentOriginala467913f9ff34913553be64599ec6e92; ?>
<?php unset($__componentOriginala467913f9ff34913553be64599ec6e92); ?>
<?php endif; ?>

                    <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->currentPaymentFields; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $field): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                        <?php
                            $fieldName = $field['name'];
                            $isTextarea = $field['type'] === 'textarea';
                        ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($field['type'] === 'text'): ?>
                            <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'newBeneficiary.'.e($fieldName).'','label' => ''.e($field['label']).'','type' => 'text']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'newBeneficiary.'.e($fieldName).'','label' => ''.e($field['label']).'','type' => 'text']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
                        <?php elseif($isTextarea): ?>
                            <?php if (isset($component)) { $__componentOriginal0ee30026125d1a66523211147b00e4dc = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal0ee30026125d1a66523211147b00e4dc = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::textarea','data' => ['wire:model.defer' => 'newBeneficiary.'.e($fieldName).'','label' => ''.e($field['label']).'','class' => 'col-span-2 w-full']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::textarea'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'newBeneficiary.'.e($fieldName).'','label' => ''.e($field['label']).'','class' => 'col-span-2 w-full']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal0ee30026125d1a66523211147b00e4dc)): ?>
<?php $attributes = $__attributesOriginal0ee30026125d1a66523211147b00e4dc; ?>
<?php unset($__attributesOriginal0ee30026125d1a66523211147b00e4dc); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal0ee30026125d1a66523211147b00e4dc)): ?>
<?php $component = $__componentOriginal0ee30026125d1a66523211147b00e4dc; ?>
<?php unset($__componentOriginal0ee30026125d1a66523211147b00e4dc); ?>
<?php endif; ?>
                        <?php elseif($field['type'] === 'select'): ?>
                            <?php if (isset($component)) { $__componentOriginala467913f9ff34913553be64599ec6e92 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginala467913f9ff34913553be64599ec6e92 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::select.index','data' => ['wire:model' => 'newBeneficiary.'.e($fieldName).'','label' => ''.e($field['label']).'']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model' => 'newBeneficiary.'.e($fieldName).'','label' => ''.e($field['label']).'']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                                <option value="" disabled>Select <?php echo e($field['label']); ?></option>
                                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $field['options'] ?? []; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $option): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                                    <option value="<?php echo e($option['value']); ?>">
                                        <?php echo e($option['label']); ?>

                                    </option>
                                <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginala467913f9ff34913553be64599ec6e92)): ?>
<?php $attributes = $__attributesOriginala467913f9ff34913553be64599ec6e92; ?>
<?php unset($__attributesOriginala467913f9ff34913553be64599ec6e92); ?>
<?php endif; ?>
<?php if (isset($__componentOriginala467913f9ff34913553be64599ec6e92)): ?>
<?php $component = $__componentOriginala467913f9ff34913553be64599ec6e92; ?>
<?php unset($__componentOriginala467913f9ff34913553be64599ec6e92); ?>
<?php endif; ?>
                        <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                    <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>

            </div>

            <div class="mt-4 flex justify-end gap-2">
                <button type="button" @click="open = false; $wire.closeModal()"
                    class="px-3 py-1 rounded bg-gray-200 dark:bg-gray-600">
                    Cancel
                </button>

                <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['type' => 'button','wire:click' => 'addBeneficiary','wire:loading.attr' => 'disabled','wire:target' => 'addBeneficiary','class' => 'px-3 py-1 rounded bg-sky-600 text-white disabled:opacity-50']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['type' => 'button','wire:click' => 'addBeneficiary','wire:loading.attr' => 'disabled','wire:target' => 'addBeneficiary','class' => 'px-3 py-1 rounded bg-sky-600 text-white disabled:opacity-50']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                    Save Beneficiary
                 <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
            </div>
        </div>
    </div>

</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/pages/user/transfer/⚡beneficiaries-card.blade.php ENDPATH**/ ?>