<?php

use Livewire\Component;
use App\Models\User;
use App\Models\Account;
use App\Models\Transaction;
use Illuminate\Support\Facades\Cache;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;

new #[Layout('layouts::base')] class extends Component {
    public User $user;

    public ?Account $selectedAccount = null;
    public ?int $selectedAccountId = null;
    public $accounts;
    public string $currencySymbol = '$';

    public array $countries = [];
    public array $states = [];
    public array $currencies = [];
    public $latestTransactions = [];
    public int $profileVersion = 0;

    // Stats
    public array $stats = [
        'walletBalance' => 0,
        'totalDeposit' => 0,
        'totalWithdrawal' => 0,
        'totalLoan' => 0,
        'revenue' => 0,
        'expenses' => 0,
        'ticketsOpen' => 0,
        'ticketsClosed' => 0,
    ];

    public function mount(User $user)
    {
        $this->user = $user;
        $this->accounts = $user->profile?->accounts ?? collect();

        // Default to primary or first account
        $primary = $this->accounts->firstWhere('is_primary', true) ?? $this->accounts->first();
        $this->selectedAccount = $primary;
        $this->selectedAccountId = $primary?->id;

        $this->currencySymbol = \App\Support\Currency::symbol($user->profile?->currency ?? 'USD');
        $this->loadCurrencies();
        // Load initial stats
        $this->loadStats();
        $this->updateLatestTransactions();
    }

    public function updatedSelectedAccountId($accountId)
    {
        $this->selectedAccount = $this->accounts->firstWhere('id', $accountId) ?? null;
        $this->loadStats(); // always reload stats when account changes
        $this->updateLatestTransactions();
    }

    private function updateLatestTransactions()
    {
        $account = $this->selectedAccount;

        if (!$account) {
            $this->latestTransactions = [];
            return;
        }

        $query = Transaction::where('account_id', $account->id);

        $this->latestTransactions = $query->latest()->take(10)->get();
    }

    public function loadCurrencies()
    {
        $this->currencies = Cache::rememberForever('currencies_json', function () {
            return collect(json_decode(file_get_contents(resource_path('data/currencies.json')), true))
                ->sortBy('name')
                ->values()
                ->toArray();
        });
    }

    public function loadStats()
    {
        // Reset
        $this->stats = [
            'walletBalance' => 0,
            'totalDeposit' => 0,
            'totalWithdrawal' => 0,
            'totalLoan' => 0,
            'revenue' => 0,
            'expenses' => 0,
            'ticketsOpen' => 0,
            'ticketsClosed' => 0,
        ];

        if (!$this->selectedAccount) {
            return;
        }

        $account = $this->selectedAccount;

        $this->stats = [
            'walletBalance' => (float) ($account->available_balance ?? 0),
            'totalDeposit' => (float) ($account->deposits()->sum('amount') ?? 0),
            'totalWithdrawal' => (float) ($account->withdrawals()->sum('amount') ?? 0),
            'totalLoan' => (float) ($account->loans()->sum('amount') ?? 0),
            'revenue' => (float) ($account->transactions()->incoming()->sum('amount') ?? 0),
            'expenses' => (float) ($account->transactions()->outgoing()->sum('amount') ?? 0),
            'ticketsOpen' => $this->user->tickets()->open()->count() ?? 0,
            'ticketsClosed' => $this->user->tickets()->closed()->count() ?? 0,
        ];
    }

    // Helper method so children can trigger stats reload
    protected $listeners = [
        'updated' => 'refreshStats',
    ];

    public function refreshStats()
    {
        $this->loadStats();
    }

    #[On('profile-updated')]
    public function handleProfileUpdated($userId)
    {
        $this->user = User::with('profile.accounts')->find($userId);
        $this->selectedAccount = $this->accounts->firstWhere('id', $this->selectedAccountId);

        $this->currencySymbol = \App\Support\Currency::symbol($this->user->profile?->currency ?? 'USD');

        $this->accounts = $this->user->profile?->accounts ?? collect();

        $this->profileVersion++; // 🔥 forces child remount
        $this->loadStats();
        $this->updateLatestTransactions();
    }
};
?>

 <?php $__env->slot('title', null, []); ?> <?php echo e(config('app.name')); ?> || Manage User: <?php echo e($user->name); ?> <?php $__env->endSlot(); ?>

<div class="py-4 space-y-5">

    <?php if (isset($component)) { $__componentOriginalf8d4ea307ab1e58d4e472a43c8548d8e = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalf8d4ea307ab1e58d4e472a43c8548d8e = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.page-header','data' => ['title' => ''.e($this->user->name).' Information','subtitle' => 'View detailed information about the user, including profile details, KYC status, and account activity.','rightText' => 'back to users','backUrl' => ''.e(route('admin.users.index')).'']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('page-header'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['title' => ''.e($this->user->name).' Information','subtitle' => 'View detailed information about the user, including profile details, KYC status, and account activity.','rightText' => 'back to users','back-url' => ''.e(route('admin.users.index')).'']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalf8d4ea307ab1e58d4e472a43c8548d8e)): ?>
<?php $attributes = $__attributesOriginalf8d4ea307ab1e58d4e472a43c8548d8e; ?>
<?php unset($__attributesOriginalf8d4ea307ab1e58d4e472a43c8548d8e); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalf8d4ea307ab1e58d4e472a43c8548d8e)): ?>
<?php $component = $__componentOriginalf8d4ea307ab1e58d4e472a43c8548d8e; ?>
<?php unset($__componentOriginalf8d4ea307ab1e58d4e472a43c8548d8e); ?>
<?php endif; ?>


    
    <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.live' => 'selectedAccountId','label' => 'Select Account','placeholder' => 'Choose Account']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.live' => 'selectedAccountId','label' => 'Select Account','placeholder' => 'Choose Account']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $accounts; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $account): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
            <option value="<?php echo e($account->id); ?>">
                <?php echo e($account->bank_account_type); ?> - <?php echo e($account->masked_account_number); ?>

                <?php echo e($currencySymbol); ?><?php echo e(number_format($account->available_balance, 2)); ?>

            </option>
        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>

    
    <?php
$__split = function ($name, $params = []) {
    return [$name, $params];
};
[$__name, $__params] = $__split('user.show.user-stats', ['stats' => $stats,'currency-symbol' => $currencySymbol]);

$key = 'stats-'.$selectedAccountId.'-'.$profileVersion;
$__componentSlots = [];

$key ??= \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::generateKey('lw-1412007177-8', $key);

$__html = app('livewire')->mount($__name, $__params, $key, $__componentSlots);

echo $__html;

unset($__html);
unset($__name);
unset($__params);
unset($__componentSlots);
unset($__split);
if (isset($__slots)) unset($__slots);
?>

    
    <?php
$__split = function ($name, $params = []) {
    return [$name, $params];
};
[$__name, $__params] = $__split('user.show.user-action', ['user' => $user,'accounts' => $accounts,'selected-account-id' => $selectedAccountId,'selected-account' => $selectedAccount,'currency-symbol' => $currencySymbol,'stats' => $stats]);

$key = 'action-'.$selectedAccountId.'-'.$profileVersion;
$__componentSlots = [];

$key ??= \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::generateKey('lw-1412007177-9', $key);

$__html = app('livewire')->mount($__name, $__params, $key, $__componentSlots);

echo $__html;

unset($__html);
unset($__name);
unset($__params);
unset($__componentSlots);
unset($__split);
if (isset($__slots)) unset($__slots);
?>

    
    <?php
$__split = function ($name, $params = []) {
    return [$name, $params];
};
[$__name, $__params] = $__split('user.show.user-kyc', ['user' => $user]);

$key = 'kyc-'.$user->id.'-'.$profileVersion;
$__componentSlots = [];

$key ??= \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::generateKey('lw-1412007177-10', $key);

$__html = app('livewire')->mount($__name, $__params, $key, $__componentSlots);

echo $__html;

unset($__html);
unset($__name);
unset($__params);
unset($__componentSlots);
unset($__split);
if (isset($__slots)) unset($__slots);
?>

    
    <?php
$__split = function ($name, $params = []) {
    return [$name, $params];
};
[$__name, $__params] = $__split('user.show.user-profile', ['user' => $user,'currency-symbol' => $currencySymbol,'currencies' => $currencies]);

$key = 'profile-'.$user->id.'-'.$profileVersion;
$__componentSlots = [];

$key ??= \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::generateKey('lw-1412007177-11', $key);

$__html = app('livewire')->mount($__name, $__params, $key, $__componentSlots);

echo $__html;

unset($__html);
unset($__name);
unset($__params);
unset($__componentSlots);
unset($__split);
if (isset($__slots)) unset($__slots);
?>



    <?php echo $__env->make('partials.recent-transaction-dashboard', [
        'latestTransactions' => $latestTransactions,
        'viewAllUrl' => route('admin.users.transactions', $user),
    ], array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>


</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/pages/admin/user/⚡show.blade.php ENDPATH**/ ?>