<?php

use Livewire\Component;
use Illuminate\Validation\Rule;
use App\Enums\LoanOptions;
use App\Support\Currency;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Transaction;
use App\Support\LoanCalculator;
use App\Notifications\TransactionNotification;
use App\Services\AdminNotifier;
new class extends Component {
    public $accounts = [];
    public $selectedAccount = null;
    public $currencySymbol;
    public $amount = '';
    public $duration = '';
    public $credit_facility = '';
    public $employment_status = '';
    public $net_income = '';
    public $purpose = '';
    public $minAmount = 300;
    public $agreement_terms = false;
    public $agreement_privacy = false;
    public $agreement_credit = false;
    public string $idempotencyKey = '';
    public string $confirmedAmount = '0';
    public string $confirmedDuration = '0';
    public string $confirmedRate = '0';
    public string $confirmedTotalInterest = '0';
    public string $confirmedMonthlyRepayment = '0';
    public ?int $confirmedAccountId = null;

    #[Computed]
    public function durations()
    {
        return LoanOptions::durations();
    }

    #[Computed]
    public function creditFacilities()
    {
        return LoanOptions::creditFacilities();
    }

    #[Computed]
    public function employmentStatuses()
    {
        return LoanOptions::employmentStatuses();
    }

    #[Computed]
    public function netIncomeRanges()
    {
        return LoanOptions::netIncomeRanges();
    }

    #[Computed]
    public function monthlyRepayment()
    {
        $amount = preg_replace('/[^\d.]/', '', $this->amount);
        $duration = preg_replace('/[^\d]/', '', $this->duration);

        if (bccomp($amount, '0', 2) <= 0 || bccomp($duration, '0', 0) <= 0 || !$this->credit_facility) {
            return '0.00';
        }

        $rate = (string) config("loan.interest_rates.{$this->credit_facility}", '0.04');

        $interestPerMonth = bcmul($amount, $rate, 4);
        $totalInterest = bcmul($interestPerMonth, $duration, 2);
        $totalRepayment = bcadd($amount, $totalInterest, 2);

        return bcdiv($totalRepayment, $duration, 2);
    }

    #[Computed]
    public function repaymentSchedule(): array
    {
        $amount = preg_replace('/[^\d.]/', '', $this->amount);
        $duration = preg_replace('/[^\d]/', '', $this->duration);

        if (bccomp($amount, '0', 2) <= 0 || bccomp($duration, '0', 0) <= 0 || !$this->credit_facility) {
            return [];
        }

        $rate = (string) config("loan.interest_rates.{$this->credit_facility}", '0.04');

        $schedule = [];

        // Calculate monthly amortized payment: M = P * [ r*(1+r)^n ] / [ (1+r)^n - 1 ]
        $r = $rate; // monthly interest rate
        $n = $duration; // number of months

        // (1+r)^n
        $onePlusR = bcadd('1', $r, 8);
        $onePlusRToN = bcpow($onePlusR, $n, 8);

        // numerator: P * r * (1+r)^n
        $numerator = bcmul(bcmul($amount, $r, 8), $onePlusRToN, 8);

        // denominator: (1+r)^n - 1
        $denominator = bcsub($onePlusRToN, '1', 8);

        // Monthly payment
        $monthlyPayment = bcdiv($numerator, $denominator, 2);

        $remainingPrincipal = $amount;

        for ($i = 1; $i <= (int) $duration; $i++) {
            // Interest for this month = remaining principal * rate
            $interest = bcmul($remainingPrincipal, $r, 2);

            // Principal = monthly payment – interest
            $principal = bcsub($monthlyPayment, $interest, 2);

            // Update remaining principal
            $remainingPrincipal = bcsub($remainingPrincipal, $principal, 2);

            // Correct last month rounding difference
            if ($i === (int) $duration && bccomp($remainingPrincipal, '0', 2) !== 0) {
                $principal = bcadd($principal, $remainingPrincipal, 2);
                $monthlyPayment = bcadd($principal, $interest, 2);
                $remainingPrincipal = '0.00';
            }

            $schedule[] = [
                'month' => $i,
                'principal' => $principal,
                'interest' => $interest,
                'total' => $monthlyPayment,
            ];
        }

        return $schedule;
    }

    /* =========================
    |  Lifecycle
    ========================= */
    public function mount()
    {
        $user = auth()->user();

        $this->accounts = $user->profile ? $user->profile->accounts()->where('is_active', true)->get() : collect();

        $this->selectedAccount = $this->accounts->first()?->id;

        $this->currencySymbol = $user->profile ? Currency::symbol($user->profile->currency) : '$';

        $this->idempotencyKey = (string) Str::uuid();
    }

    protected function rules(): array
    {
        return [
            'amount' => 'required|numeric',
            'duration' => ['required', Rule::in(LoanOptions::durations())],
            'credit_facility' => ['required', Rule::in(LoanOptions::creditFacilities())],
            'employment_status' => ['required', Rule::in(LoanOptions::employmentStatuses())],
            'net_income' => ['required', Rule::in(LoanOptions::netIncomeRanges())],
            'purpose' => 'required|string',
            'agreement_terms' => 'accepted',
            'agreement_privacy' => 'accepted',
            'agreement_credit' => 'accepted',
        ];
    }

    public function confirmLoan()
    {
        $this->validate();

        $user = auth()->user();
        $account = $this->accounts->firstWhere('id', $this->selectedAccount);

        if (!$user->kyc_verified) {
            $this->addError('selectedAccount', 'Complete KYC before initiating a loan.');
            return;
        }

        if (!$account) {
            $this->addError('selectedAccount', 'Invalid account selected.');
            return;
        }

        if (\App\Models\Loan::where('idempotency_key', $this->idempotencyKey)->exists()) {
            $this->addError('selectedAccount', 'This loan has already been submitted.');
            return;
        }

        // -------------------------------
        // Amount cleanup (string)
        // -------------------------------
        $amount = preg_replace('/[^\d.]/', '', $this->amount);
        if (bccomp($amount, '0', 2) <= 0) {
            $this->addError('amount', 'Invalid amount.');
            return;
        }

        $minAmount = (string) config('loan.min_amount', '300');
        if (bccomp($amount, $minAmount, 2) === -1) {
            $this->addError('amount', "Minimum loan amount is {$this->currencySymbol}{$minAmount}.");
            return;
        }

        // -------------------------------
        // Income eligibility
        // -------------------------------
        $netIncome = LoanCalculator::parseNetIncome($this->net_income); // MUST return string
        if (bccomp($netIncome, '0', 2) <= 0) {
            $this->addError('net_income', 'Invalid net income value.');
            return;
        }
        $incomeRanges = LoanOptions::netIncomeRanges(); // assign to variable
        $lastRange = end($incomeRanges); // now it works

        // Skip max loan check if user selected the last range (open-ended)
        if ($this->net_income !== $lastRange) {
            $maxLoan = LoanCalculator::maxLoan($netIncome); // string

            // Format nicely with commas and 2 decimals
            $formattedMaxLoan = 'Bs' . number_format((float) $maxLoan, 2);

            if (bccomp($amount, $maxLoan, 2) === 1) {
                $this->addError('amount', "Maximum allowed loan based on your income is {$formattedMaxLoan}.");
                return;
            }
        }

        // -------------------------------
        // Duration + rate
        // -------------------------------
        $duration = preg_replace('/[^\d]/', '', $this->duration);
        if (bccomp($duration, '0', 0) <= 0) {
            $this->addError('duration', 'Invalid loan duration.');
            return;
        }

        $rate = (string) config("loan.interest_rates.{$this->credit_facility}", '0.04');

        // -------------------------------
        // Generate repayment schedule
        // -------------------------------
        $schedule = $this->repaymentSchedule;

        if (empty($schedule)) {
            $this->addError('amount', 'Cannot generate repayment schedule.');
            return;
        }

        // Compute totals from schedule
        $totalPrincipal = '0';
        $totalInterest = '0';
        foreach ($schedule as $item) {
            $totalPrincipal = bcadd($totalPrincipal, $item['principal'], 2);
            $totalInterest = bcadd($totalInterest, $item['interest'], 2);
        }

        $totalRepayment = bcadd($totalPrincipal, $totalInterest, 2);

        // Take first month total as representative monthly repayment
        $monthlyRepayment = $schedule[0]['total'];

        // -------------------------------
        // Persist confirmation state
        // -------------------------------
        $this->confirmedAmount = $amount;
        $this->confirmedDuration = $duration;
        $this->confirmedRate = $rate;
        $this->confirmedTotalInterest = $totalInterest;
        $this->confirmedMonthlyRepayment = $monthlyRepayment;
        $this->confirmedAccountId = $account->id;

        // Open modal
        $this->dispatch('open-loan-confirmation');
    }

    public function applyLoan()
    {
        $user = auth()->user();
        $account = $this->accounts->firstWhere('id', $this->confirmedAccountId);

        if (!$account || bccomp($this->confirmedAmount, '0', 2) <= 0) {
            $this->addError('amount', 'Invalid loan confirmation state.');
            return;
        }

        try {
            DB::beginTransaction();

            $loan = \App\Models\Loan::create([
                'user_id' => $user->id,
                'idempotency_key' => $this->idempotencyKey,
                'account_id' => $account->id,
                'amount' => $this->confirmedAmount,
                'duration' => $this->confirmedDuration,
                'credit_facility' => $this->credit_facility,
                'purpose' => $this->purpose,
                'interest_rate' => $this->confirmedRate,
                'total_interest' => $this->confirmedTotalInterest,
                'monthly_repayment' => $this->confirmedMonthlyRepayment,
                'principal_outstanding' => $this->confirmedAmount, // full principal initially
                'interest_outstanding' => $this->confirmedTotalInterest, // full interest initially
                'status' => 'pending',
                'first_due_date' => now()->addMonth(),
                'last_due_date' => now()->addMonths((int) $this->confirmedDuration),
            ]);

            $transaction = Transaction::create([
                'user_id' => $user->id,
                'idempotency_key' => $this->idempotencyKey,
                'account_id' => $account->id,
                'loan_id' => $loan->id,
                'type' => 'loan',
                'amount' => $this->confirmedAmount,
                'status' => Transaction::STATUS_PENDING,
                'balance_after' => $account->available_balance,
                'description' => 'Loan disbursement',
            ]);

            $user->notify(new TransactionNotification($transaction));

            AdminNotifier::notify(
                'Loan Request Submitted',
                [
                    'loan_id' => $loan->id,
                    'account_id' => $account->id,
                    'amount' => $transaction->amount,
                ],
                $user,
            );

            DB::commit();

            // Reset idempotency AFTER success
            $this->idempotencyKey = (string) Str::uuid();
            $this->dispatch('loan-success');
            $this->dispatch('transaction-success', type: 'loan', amount: $this->confirmedAmount);
            $this->dispatch('notification-sent');
            $this->reset(['amount', 'duration', 'credit_facility', 'employment_status', 'net_income', 'purpose', 'confirmedAmount', 'confirmedDuration', 'confirmedRate', 'confirmedTotalInterest', 'confirmedMonthlyRepayment', 'confirmedAccountId']);
        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Loan application failed', ['error' => $e]);
            $this->addError('amount', 'Something went wrong while processing your loan.');
        }
    }
};
?>

<x-slot:title>
    {{ config('app.name') }} || Loan Application
</x-slot:title>

<div class="space-y-5 py-4 font-sans">
    <div class="flex items-center justify-between gap-2 flex-wrap">
        <a href="{{ route('user.loans') }}"
            class="bg-sky-600 text-white flex items-center hover:shadow-2xl justify-center gap-1 px-3 py-2 rounded-md">
            <x-flux::icon.arrow-left variant="solid" class="w-4 h-4" />
            <span class="text-sm font-medium">Back to Information</span>
        </a>

        <a href="{{ route('user.loans.index') }}"
            class="bg-sky-600 text-white flex items-center hover:shadow-2xl justify-center gap-1 px-3 py-2 rounded-md">
            <x-flux::icon.hand-coins class="w-4 h-4" />
            <span class="text-sm font-medium">View Loans History</span>
        </a>
    </div>

    <div class="bg-white dark:bg-gray-900 border border-gray-100 dark:border-gray-800 rounded-xl p-4 md:p-6 space-y-4">
        <h2 class="flex items-center gap-3 text-sm md:text-base font-semibold text-gray-800 dark:text-gray-100">
            <span
                class="bg-sky-500/10 dark:bg-sky-400/10 w-9 h-9 flex items-center justify-center rounded-full text-sky-600 dark:text-sky-400">
                <x-flux::icon.banknotes variant="solid" class="w-5 h-5" />
            </span>
            Loan Details
        </h2>

        <form wire:submit.prevent="confirmLoan" class="space-y-6 p-4 bg-white dark:bg-gray-800 rounded-lg shadow-sm">
            <div class="flex items-center justify-between gap-3 flex-wrap">
                <h3 class="text-lg font-bold text-gray-900 dark:text-gray-100 mb-4">
                    Loan Application
                </h3>
                <span class="text-red-400 text-sm font-medium capitalize">* all fields are required</span>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>

                    {{-- Loan Account --}}

                    {{-- Select Account --}}
                    <x-select wire:model.defer="selectedAccount" label="Select Account"
                        placeholder="Select Account You're Applying Loan On">
                        @foreach ($accounts as $account)
                            <option value="{{ $account->id }}">
                                {{ $account->bank_account_type }} - {{ $account->masked_account_number }} -
                                {{ $currencySymbol }}{{ $account->available_balance }}
                            </option>
                        @endforeach
                    </x-select>
                </div>
                {{-- Loan Amount --}}
                <div>
                    <x-input wire:model.defer="amount" label="Loan Amount" type="number" step="0.01"
                        placeholder="Enter desired loan amount" required />
                </div>

                {{-- Loan Duration --}}
                <div>
                    <x-select wire:model.defer="duration" label="Loan Duration" placeholder="Select duration" required>
                        @foreach ($this->durations as $d)
                            <option value="{{ $d }}">{{ $d }} Months</option>
                        @endforeach
                    </x-select>
                </div>

                {{-- Credit Facility --}}
                <div>
                    <x-select wire:model.defer="credit_facility" label="Credit Facility" placeholder="Select type"
                        required>
                        @foreach ($this->creditFacilities as $cf)
                            <option value="{{ $cf }}">{{ ucfirst($cf) }} Loan</option>
                        @endforeach
                    </x-select>
                </div>

                {{-- Financial Information Section --}}
                <div class="md:col-span-2 mt-4">
                    <h4 class="flex items-center gap-3 text-base font-semibold text-gray-800 dark:text-gray-100 mb-2">
                        <span
                            class="bg-sky-500/10 dark:bg-sky-400/10 w-9 h-9 flex items-center justify-center rounded-full text-sky-600 dark:text-sky-400">
                            <x-flux::icon.wallet variant="solid" class="w-5 h-5" />
                        </span>
                        Financial Information
                    </h4>
                </div>

                {{-- Net Income --}}
                <div>
                    <x-select wire:model.defer="net_income" label="Monthly Net Income" placeholder="Select range"
                        required>
                        @foreach ($this->netIncomeRanges as $range)
                            <option value="{{ $range }}">{{ $range }}</option>
                        @endforeach
                    </x-select>
                </div>

                {{-- Employment Status --}}
                <div>
                    <x-select wire:model.defer="employment_status" label="Employment Status" placeholder="Select status"
                        required>
                        @foreach ($this->employmentStatuses as $es)
                            <option value="{{ $es }}">{{ ucwords(str_replace('-', ' ', $es)) }}</option>
                        @endforeach
                    </x-select>
                </div>

                {{-- Loan Purpose --}}
                <div class="md:col-span-2">
                    <x-textarea wire:model.defer="purpose" label="Loan Purpose"
                        placeholder="Briefly describe the purpose of the loan" rows="3" required />
                </div>

                {{-- Agreement Checkboxes --}}
                <div
                    class="md:col-span-2 mt-4 space-y-3 p-4 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                    <h4 class="text-gray-800 dark:text-gray-100 font-semibold mb-2">Agreements & Consent</h4>

                    {{-- Terms & Conditions --}}
                    <div class="flex items-start gap-2">
                        <input type="checkbox" wire:model.defer="agreement_terms" id="agreement_terms" required
                            class="mt-1 rounded border-gray-300 text-sky-600 focus:ring-sky-500">
                        <label for="agreement_terms" class="text-gray-700 dark:text-gray-200 text-sm">
                            I agree to the <a href="#" class="text-sky-600 hover:underline">Terms &
                                Conditions</a>.
                        </label>
                    </div>

                    {{-- Privacy Policy --}}
                    <div class="flex items-start gap-2">
                        <input type="checkbox" wire:model.defer="agreement_privacy" id="agreement_privacy" required
                            class="mt-1 rounded border-gray-300 text-sky-600 focus:ring-sky-500">
                        <label for="agreement_privacy" class="text-gray-700 dark:text-gray-200 text-sm">
                            I acknowledge the <a href="#" class="text-sky-600 hover:underline">Privacy Policy</a>.
                        </label>
                    </div>

                    {{-- Credit Check Authorization --}}
                    <div class="flex items-start gap-2">
                        <input type="checkbox" wire:model.defer="agreement_credit" id="agreement_credit" required
                            class="mt-1 rounded border-gray-300 text-sky-600 focus:ring-sky-500">
                        <label for="agreement_credit" class="text-gray-700 dark:text-gray-200 text-sm">
                            I authorize {{ config('app.name') }} to verify my information and credit history.
                        </label>
                    </div>

                    {{-- Confirmation Text --}}
                    <p class="text-gray-600 dark:text-gray-300 text-sm mt-2">
                        By submitting this application, I confirm that all information provided is accurate and
                        complete.
                    </p>
                </div>

                {{-- Trust Badges / Security Shields --}}
                <div class="md:col-span-2 flex items-center gap-4 mt-4">
                    <div class="flex items-center gap-2 text-sm text-gray-500 dark:text-gray-400">
                        <x-flux::icon.shield-check class="w-5 h-5 text-green-500" /> Secure & Encrypted
                    </div>
                    <div class="flex items-center gap-2 text-sm text-gray-500 dark:text-gray-400">
                        <x-flux::icon.lock-closed class="w-5 h-5 text-blue-500" /> Trusted Platform
                    </div>
                    <div class="flex items-center gap-2 text-sm text-gray-500 dark:text-gray-400">
                        <x-flux::icon.credit-card class="w-5 h-5 text-yellow-500" /> Protected Transactions
                    </div>
                </div>
            </div>

            {{-- Submit Button --}}
            <div class="mt-6">
                <x-button type="submit" class="w-full">
                    Apply for Loan
                </x-button>
            </div>
        </form>

    </div>

    <!-- CONFIRMATION MODAL -->
    <!-- CONFIRMATION MODAL -->
    <x-modal maxWidth="max-w-lg" @open-loan-confirmation.window="openModal()" @close-loan-confirmation.window="close()"
        x-on:loan-success.window="close()">

        <div class="space-y-6">

            <!-- Header -->
            <div class="text-center space-y-3">
                <div
                    class="w-14 h-14 bg-linear-to-br from-sky-500 to-sky-800 rounded-2xl mx-auto flex items-center justify-center text-white shadow-lg">
                    <x-flux::icon.banknotes variant="solid" class="w-6 h-6" />
                </div>

                <h2 class="text-lg font-bold text-gray-900 dark:text-gray-100">
                    Confirm Loan Application
                </h2>

                <p class="text-sm text-gray-500 dark:text-gray-300 max-w-xs mx-auto">
                    Please review your loan details carefully before submitting.
                </p>
            </div>

            <!-- Summary -->
            @php
                // Compute totals from the repayment schedule
                $totalPrincipal = 0;
                $totalInterest = 0;
                foreach ($this->repaymentSchedule as $item) {
                    $totalPrincipal += (float) $item['principal'];
                    $totalInterest += (float) $item['interest'];
                }
                $totalRepayment = $totalPrincipal + $totalInterest;
            @endphp

            <div class="bg-gray-50 dark:bg-gray-800 rounded-xl p-4 border border-gray-200 dark:border-gray-700">
                <ul class="space-y-2 text-sm text-gray-700 dark:text-gray-200">

                    <li class="flex justify-between">
                        <span class="font-medium">Account</span>
                        <span>{{ $accounts->firstWhere('id', $selectedAccount)?->masked_account_number ?? '-' }}</span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Loan Amount</span>
                        <span>{{ $currencySymbol }}{{ number_format((float) $confirmedAmount, 2) }}</span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Duration</span>
                        <span>{{ $confirmedDuration }} months</span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Credit Facility</span>
                        <span>{{ ucfirst($credit_facility) }} Loan</span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Monthly Repayment</span>
                        <span class="font-semibold text-sky-600 dark:text-sky-400">
                            {{ $currencySymbol }}{{ number_format((float) $confirmedMonthlyRepayment, 2) }}
                        </span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Total Interest</span>
                        <span>{{ $currencySymbol }}{{ number_format($totalInterest, 2) }}</span>
                    </li>

                    <li class="flex justify-between border-t pt-2 mt-2">
                        <span class="font-semibold">Total Repayment</span>
                        <span class="font-bold">
                            {{ $currencySymbol }}{{ number_format($totalRepayment, 2) }}
                        </span>
                    </li>
                </ul>
            </div>

            <!-- Repayment Schedule -->
            <div class="space-y-2">
                <h4 class="font-semibold text-gray-800 dark:text-gray-100">
                    Repayment Schedule
                </h4>

                <div class="overflow-hidden rounded-lg border border-gray-200 dark:border-gray-700">
                    <table class="w-full text-sm">
                        <thead class="bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                            <tr>
                                <th class="px-3 py-2 text-left">Month</th>
                                <th class="px-3 py-2 text-right">Principal</th>
                                <th class="px-3 py-2 text-right">Interest</th>
                                <th class="px-3 py-2 text-right">Total</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white dark:bg-gray-800">
                            @php
                                $runningPrincipal = 0;
                                $runningInterest = 0;
                            @endphp
                            @foreach ($this->repaymentSchedule as $item)
                                @php
                                    $principal = number_format((float) $item['principal'], 2);
                                    $interest = number_format((float) $item['interest'], 2);
                                    $total = number_format((float) $item['total'], 2);

                                    $runningPrincipal += (float) $item['principal'];
                                    $runningInterest += (float) $item['interest'];
                                @endphp

                                <tr class="border-t border-gray-200 dark:border-gray-700">
                                    <td class="px-3 py-2">{{ $item['month'] }}</td>
                                    <td class="px-3 py-2 text-right">{{ $currencySymbol }}{{ $principal }}</td>
                                    <td class="px-3 py-2 text-right">{{ $currencySymbol }}{{ $interest }}</td>
                                    <td class="px-3 py-2 text-right font-semibold">
                                        {{ $currencySymbol }}{{ $total }}</td>
                                </tr>
                            @endforeach

                            <!-- Totals Row -->
                            <tr
                                class="border-t border-gray-200 dark:border-gray-700 font-bold bg-gray-50 dark:bg-gray-700">
                                <td class="px-3 py-2 text-left">Total</td>
                                <td class="px-3 py-2 text-right">
                                    {{ $currencySymbol }}{{ number_format($runningPrincipal, 2) }}</td>
                                <td class="px-3 py-2 text-right">
                                    {{ $currencySymbol }}{{ number_format($runningInterest, 2) }}</td>
                                <td class="px-3 py-2 text-right">
                                    {{ $currencySymbol }}{{ number_format($runningPrincipal + $runningInterest, 2) }}
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Actions -->
            <div class="space-y-3">
                <x-button wire:click="applyLoan" variant="primary" class="w-full bg-sky-600 hover:bg-sky-700">
                    Confirm & Submit
                </x-button>

                <button type="button" @click="$dispatch('close-loan-confirmation')"
                    class="w-full text-center text-sm text-red-600 hover:underline">
                    Cancel
                </button>
            </div>
        </div>
    </x-modal>

</div>
