<?php

use Livewire\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public $trustedDevices = [];

    public function mount()
    {
        $this->loadDevices();
    }

    public function loadDevices()
    {
        $this->trustedDevices = Auth::user()->trustedDevices()->orderByDesc('last_used_at')->get();
    }

    public function revokeDevice(int $deviceId): void
    {
        $device = Auth::user()->trustedDevices()->whereKey($deviceId)->first();

        if (!$device) {
            $this->dispatch('showToast', message: 'Device not found or already revoked.', type: 'warning');

            return;
        }

        $device->delete();

        $this->dispatch('showToast', message: 'Trusted device revoked.', type: 'success');

        $this->loadDevices();
    }
};
?>

<x-slot:title>
    {{ config('app.name') }} || Manage Trusted Device's
</x-slot:title>

<section class="w-full py-5">
    <x-settings.layout :heading="__('Trusted Devices')" :subheading="__('Manage your trusted devices and revoke access')">
        <section class="md:col-span-3 bg-white dark:bg-gray-900 rounded-2xl p-6 shadow-sm space-y-6">
            @if ($trustedDevices->count())
                <div class="space-y-6">
                    @foreach ($trustedDevices as $device)
                        @php
                            $lastUsed = $device->last_used_at;
                            $isActive = $lastUsed && $lastUsed->greaterThan(now()->subDays(7));
                        @endphp

                        <div
                            class="flex items-center justify-between bg-gray-50 dark:bg-gray-800 rounded-md p-4 shadow-sm">
                            <div class="flex items-start space-x-4 grow min-w-0">
                                <div class="text-[#0E4D92] dark:text-[#F78F20] shrink-0">
                                    <x-flux::icon.device-phone-mobile class="w-7 h-7" />
                                </div>

                                <div class="truncate max-w-xs">
                                    <p class="font-semibold text-gray-900 dark:text-gray-100 truncate">
                                        {{ $device->device_type }}
                                        —
                                        <span class="font-normal">
                                            {{ $device->browser }} on {{ $device->platform }}
                                        </span>
                                    </p>

                                    <div class="mt-2 text-gray-600 dark:text-gray-400 text-xs space-y-1">
                                        <p><strong>IP:</strong> {{ $device->ip_address ?? 'Unknown' }}</p>
                                        <p>
                                            <strong>Last used:</strong>
                                            {{ $device->last_used_at?->diffForHumans() ?? 'Never used' }}
                                        </p>
                                        <p class="italic truncate">
                                            <strong>User agent:</strong>
                                            {{ $device->user_agent ?? 'Unknown' }}
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <div class="flex flex-col items-end space-y-2 ml-4 shrink-0">
                                <span
                                    class="px-3 py-0.5 rounded-full text-xs font-semibold
                                        {{ $isActive
                                            ? 'bg-green-200 text-green-800 dark:bg-green-700 dark:text-green-100'
                                            : 'bg-gray-200 text-gray-700 dark:bg-gray-600 dark:text-gray-300' }}">
                                    {{ $isActive ? 'Active' : 'Inactive' }}
                                </span>

                                <x-button variant="danger" wire:click="revokeDevice({{ $device->id }})"
                                    wire:confirm="Are you sure you want to revoke this trusted device?">
                                    Revoke device
                                </x-button>
                            </div>
                        </div>
                    @endforeach
                </div>
            @else
                <p class="text-center text-gray-600 dark:text-gray-400 italic mt-6">
                    You have no trusted devices stored.
                </p>
            @endif
        </section>
    </x-settings.layout>
</section>
