<?php

use Livewire\Component;
use App\Models\Loan;
use Livewire\Attributes\Computed;
use App\Models\LoanInstallment;
use Illuminate\Support\Facades\DB;
use App\Models\Transaction;
use App\Notifications\TransactionNotification;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Traits\WithTableHelpers;

new class extends Component {
    use WithPagination, WithTableHelpers;
    public Loan $loan;
    public string $currencySymbol = '$';
    public ?int $selectedAccountId = null;
    public $userAccounts = [];

    public $perPage = 10;
    public int $page = 1;

    protected function searchableFields(): array
    {
        return ['installment_number', 'due_date', 'principal', 'interest', 'penalty', 'total', 'status', 'paid_at'];
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->dispatch('perPageUpdated', $this->perPage);
    }

    public function mount(Loan $loan)
    {
        $this->loan = $loan;

        $user = auth()->user();
        $this->currencySymbol = $user->profile ? \App\Support\Currency::symbol($user->profile->currency) : '$';

        // Load all user accounts
        $this->userAccounts = $user->profile?->accounts()->get() ?? collect();
        $this->selectedAccountId = $this->userAccounts->first()?->id; // default first account
    }

    /**
     * Pay a specific installment using user's account balance
     */
    public function payInstallment($installmentId)
    {
        $installment = LoanInstallment::findOrFail($installmentId);
        $user = auth()->user();

        // Find the account selected by the user
        $account = $this->userAccounts->firstWhere('id', $this->selectedAccountId);
        if (!$account) {
            $this->dispatch('showToast', message: 'Please select an account to pay from.', type: 'error');
            return;
        }

        if ($installment->status !== 'pending') {
            $this->dispatch('showToast', message: 'This installment is already paid or not payable.', type: 'error');
            return;
        }

        try {
            DB::transaction(function () use ($installment, $user, $account) {
                $loan = $this->loan->fresh();
                $account = $account->fresh(); // Lock account for update

                // Check balance
                if (bccomp($account->balance, $installment->total, 2) === -1) {
                    throw new \Exception('Insufficient balance in the selected account.');
                }

                // Deduct balance
                $account->balance = bcsub($account->balance, $installment->total, 2);
                $account->save();

                // Mark installment as paid
                $installment->update([
                    'status' => 'paid',
                    'paid_at' => now(),
                ]);

                // Update loan balances
                $loan->principal_outstanding = bcsub($loan->principal_outstanding, $installment->principal, 2);
                $loan->interest_outstanding = bcsub($loan->interest_outstanding, $installment->interest, 2);
                $loan->penalty_outstanding = bcsub($loan->penalty_outstanding, $installment->penalty, 2);
                $loan->total_paid = bcadd($loan->total_paid, $installment->total, 2);
                $loan->save();
                $loan->markCompletedIfPaid();

                // Create transaction record including balance_after
                $transaction = Transaction::create([
                    'user_id' => $user->id,
                    'account_id' => $account->id,
                    'type' => 'loan',
                    'amount' => $installment->total,
                    'balance_after' => $account->balance, // ✅ store balance after deduction
                    'description' => 'Loan installment payment',
                    'status' => Transaction::STATUS_COMPLETED,
                ]);

                // Send notification
                $user->notify(new TransactionNotification($transaction));
            });

            $this->dispatch('showToast', message: 'Payment Successful.', type: 'success');
            $this->dispatch('refresh-installment');
            $this->dispatch('notification-sent');
        } catch (\Exception $e) {
            $this->dispatch('showToast', message: $e->getMessage(), type: 'error');
        }
    }

    #[On('refresh-installment')]
    #[Computed]
    public function installments()
    {
        $query = $this->loan->installments()->orderBy('installment_number');

        $this->applySearch($query, $this->searchableFields());

        return $query->orderBy($this->sortField, $this->sortDirection)->paginate($this->perPage)->withQueryString();
    }
};
?>

<div class="space-y-6 py-6">

    {{-- Loan Info --}}
    <div class="bg-white dark:bg-gray-800 p-6 rounded-lg shadow">
        <h2 class="text-xl font-semibold mb-4">Loan Details</h2>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-gray-700 dark:text-gray-300">
            <div><strong>Loan ID:</strong> {{ $loan->id }}</div>
            <div><strong>Purpose:</strong> {{ $loan->purpose ?? '-' }}</div>
            <div><strong>Amount:</strong> {{ $currencySymbol }}{{ number_format($loan->amount, 2) }}</div>
            <div><strong>Total Interest:</strong> {{ $currencySymbol }}{{ number_format($loan->total_interest, 2) }}
            </div>
            <div><strong>Monthly Repayment:</strong>
                {{ $currencySymbol }}{{ number_format($loan->monthly_repayment, 2) }}</div>
            <div><strong>Total Repayment:</strong> {{ $currencySymbol }}{{ number_format($loan->total_repayment, 2) }}
            </div>
            <div><strong>Remaining Balance:</strong>
                {{ $currencySymbol }}{{ number_format($loan->remaining_balance, 2) }}</div>
            <div><strong>Status:</strong>
                <span
                    class="px-2 py-1 rounded-full text-xs {{ match ($loan->status) {
                        'approved' => 'bg-green-500 text-white',
                        'pending' => 'bg-yellow-400 text-black',
                        'rejected' => 'bg-red-500 text-white',
                        'overdue' => 'bg-red-600 text-white',
                        'completed' => 'bg-blue-500 text-white',
                        'defaulted' => 'bg-gray-500 text-white',
                        default => 'bg-gray-300 text-black',
                    } }}">
                    {{ ucfirst($loan->status) }}
                </span>
            </div>
            <div><strong>First Due Date:</strong> {{ $loan->first_due_date?->format('d M Y') ?? '-' }}</div>
            <div><strong>Last Due Date:</strong> {{ $loan->last_due_date?->format('d M Y') ?? '-' }}</div>
        </div>
    </div>
    <div>
        <x-select wire:model="selectedAccountId" label="Select Account" placeholder="Select Account">
            @foreach ($userAccounts as $account)
                <option value="{{ $account->id }}">
                    {{ $account->bank_account_type }} - {{ $account->masked_account_number }} -
                    {{ $currencySymbol }}{{ number_format($account->available_balance, 2) }}
                </option>
            @endforeach
        </x-select>
    </div>
    <x-table.wrapper :items="$this->installments">

        {{-- SEARCH --}}
        <x-slot name="search">
            <div class="flex items-center justify-between gap-4">
                <div class="w-full md:w-1/3">
                    <div class="relative">
                        <x-flux::icon.magnifying-glass
                            class="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400 dark:text-gray-500" />
                        <input type="text" wire:model.live="search" placeholder="Search by name or email"
                            class="w-full pl-10 pr-4 py-2 text-sm rounded-md bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-sky-600 focus:border-sky-600 transition" />
                    </div>
                </div>
            </div>
        </x-slot>
        {{-- ================= HEADERS ================= --}}
        <x-slot name="head">
            <x-table.head label="#" field="id" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Due Date" field="due_date" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Principal" field="principal" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Interest" field="interest" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Penalty" field="penalty" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Total" field="total" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Status" field="status" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Paid At" field="paid_at" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Action" />
        </x-slot>

        {{-- ================= DESKTOP ================= --}}
        <x-slot name="body">
            @forelse ($this->installments as $inst)
                @php
                    // Auto Status Color Detection
                    $statusClasses = match ($inst->status) {
                        'pending' => 'bg-yellow-100 text-yellow-800',
                        'overdue' => 'bg-red-200 text-red-900',
                        'paid' => 'bg-green-100 text-green-800',
                        default => 'bg-gray-100 text-gray-700',
                    };

                    // Highlight overdue entire row
                    $rowHighlight = $inst->status === 'overdue' ? 'bg-red-50 dark:bg-red-900/30' : '';
                @endphp

                <x-table.row wire:key="installment-{{ $inst->id }}"
                    class="{{ $rowHighlight }} hover:shadow-md transition">

                    <x-table.cell>
                        {{ $inst->installment_number }}
                    </x-table.cell>

                    <x-table.cell>
                        {{ $inst->due_date?->format('d M Y') ?? '-' }}
                    </x-table.cell>

                    <x-table.cell>
                        {{ $currencySymbol }}{{ number_format($inst->principal, 2) }}
                    </x-table.cell>

                    <x-table.cell>
                        {{ $currencySymbol }}{{ number_format($inst->interest, 2) }}
                    </x-table.cell>

                    <x-table.cell>
                        {{ $currencySymbol }}{{ number_format($inst->penalty, 2) }}
                    </x-table.cell>

                    <x-table.cell class="font-semibold">
                        {{ $currencySymbol }}{{ number_format($inst->total, 2) }}
                    </x-table.cell>

                    <x-table.cell>
                        <span class="px-2 py-1 rounded-full text-xs font-medium {{ $statusClasses }}">
                            {{ ucfirst($inst->status) }}
                        </span>
                    </x-table.cell>

                    <x-table.cell>
                        {{ $inst->paid_at?->format('d M Y H:i') ?? '-' }}
                    </x-table.cell>

                    <x-table.cell>
                        @if ($inst->status === 'pending')
                            <x-flux::button wire:click="payInstallment({{ $inst->id }})" type="button"
                                variant="primary" color="green">
                                Pay
                            </x-flux::button>
                        @else
                            -
                        @endif
                    </x-table.cell>

                </x-table.row>

            @empty
                <tr>
                    <td colspan="9" class="px-6 py-8 text-center text-gray-400 dark:text-gray-500">
                        No installments found
                    </td>
                </tr>
            @endforelse
        </x-slot>

        {{-- ================= MOBILE VERSION ================= --}}
        <x-slot name="mobile">
            @forelse ($this->installments as $inst)
                @php
                    $statusClasses = match ($inst->status) {
                        'pending' => 'bg-yellow-100 text-yellow-800',
                        'overdue' => 'bg-red-200 text-red-900',
                        'paid' => 'bg-green-100 text-green-800',
                        default => 'bg-gray-100 text-gray-700',
                    };

                    $cardHighlight = $inst->status === 'overdue' ? 'border-red-400 bg-red-50 dark:bg-red-900/30' : '';
                @endphp

                <x-table.mobile-card wire:key="mobile-installment-{{ $inst->id }}"
                    class="rounded-xl shadow-md p-4 mb-4 {{ $cardHighlight }}">

                    <div class="space-y-2 text-sm text-gray-700 dark:text-gray-300">

                        <div><strong>#:</strong> {{ $inst->installment_number }}</div>
                        <div><strong>Due:</strong> {{ $inst->due_date?->format('d M Y') ?? '-' }}</div>

                        <div><strong>Principal:</strong>
                            {{ $currencySymbol }}{{ number_format($inst->principal, 2) }}
                        </div>

                        <div><strong>Interest:</strong>
                            {{ $currencySymbol }}{{ number_format($inst->interest, 2) }}
                        </div>

                        <div><strong>Penalty:</strong>
                            {{ $currencySymbol }}{{ number_format($inst->penalty, 2) }}
                        </div>

                        <div class="font-semibold">
                            <strong>Total:</strong>
                            {{ $currencySymbol }}{{ number_format($inst->total, 2) }}
                        </div>

                        <div>
                            <strong>Status:</strong>
                            <span
                                class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium {{ $statusClasses }}">
                                {{ ucfirst($inst->status) }}
                            </span>
                        </div>

                        <div>
                            <strong>Paid At:</strong>
                            {{ $inst->paid_at?->format('d M Y H:i') ?? '-' }}
                        </div>

                    </div>

                    @if ($inst->status === 'pending')
                        <div class="mt-3">
                            @if ($inst->status === 'pending')
                                <x-flux::button wire:click="payInstallment({{ $inst->id }})" type="button"
                                    variant="primary" color="green">
                                    Pay
                                </x-flux::button>
                            @else
                                -
                            @endif
                        </div>
                    @endif

                </x-table.mobile-card>

            @empty
                <div class="px-6 py-6 text-center text-gray-400 dark:text-gray-500">
                    No installments found
                </div>
            @endforelse
        </x-slot>


        {{-- Footer (pagination + perPage dropdown) --}}
        <x-slot name="footer">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between w-full gap-4 mt-6">
                <div class="w-full md:w-auto text-center">{{ $this->installments->links() }}</div>
            </div>
        </x-slot>

    </x-table.wrapper>

</div>
