<?php

use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use App\Traits\WithTableHelpers;
use App\Services\ExportService;
use App\Models\User;

new #[Layout('layouts::base', ['title' => 'All users'])] class extends Component {
    use WithPagination, WithTableHelpers;

    public $perPage = 10;
    public int $page = 1;
    public ?string $modalMessage = null;
    public string $currencySymbol = '$';
    public array $visibility = [];

    public function clearModalMessage()
    {
        $this->modalMessage = null;
    }

    /**
     * Export transactions (download or email)
     */
    public function export()
    {
        // Get transactions based on scope
        $transactions = $this->exportScope === 'page' ? $this->transactions->getCollection() : $this->baseQuery()->get();

        // Prevent full PDF export if needed
        if ($this->exportFormat === 'pdf' && $this->exportScope === 'all') {
            $this->exportScope = 'page';
        }

        // Show warning if no data
        if ($transactions->isEmpty()) {
            $this->dispatch('showToast', message: 'No data to export.', type: 'warning');
            return;
        }

        // Export via download
        if ($this->exportDelivery === 'download') {
            return $this->exportService()->download($transactions, $this->exportFormat, 'transactions', pdfTitle: 'Transactions Export');
        }

        // Export via email
        $this->exportService()->email($transactions, $this->exportFormat, 'transactions', auth()->user()->email, pdfTitle: 'Transactions Export');

        $this->dispatch('showToast', message: 'Export emailed successfully.', type: 'success');
    }

    protected function exportService(): ExportService
    {
        return app(ExportService::class);
    }

    protected function searchableFields(): array
    {
        return ['id', 'email', 'name', 'phone', 'country', 'state', 'city', 'address'];
    }

    public function delete($id)
    {
        User::findOrFail($id)->delete();
        $this->dispatch('showToast', message: 'User successfully deleted', type: 'success');
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->dispatch('perPageUpdated', $this->perPage);
    }

    #[Computed]
    public function users()
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        // Eager load profile and accounts to prevent N+1 queries
        $query = User::with(['profile', 'accounts']);

        // Apply search across searchable fields
        $this->applySearch($query, $this->searchableFields());

        // Apply visibility filter if admin exists
        if ($admin) {
            $query->visibleToAdmin($admin);
        }

        // Order and paginate
        $paginated = $query->orderBy($this->sortField, $this->sortDirection)->paginate($this->perPage)->withQueryString();

        // Populate visibility for the current page
        foreach ($paginated as $user) {
            $this->visibility[$user->id] = $user->visibility_level;
        }

        return $paginated;
    }

    public function updateVisibility($userId, $level)
    {
        $admin = auth('admin')->user();
        $user = User::withoutGlobalScope('visibility')->findOrFail($userId);

        if (!$admin || !$admin->can('updateVisibility', $user)) {
            abort(403);
        }

        $user->visibility_level = $level;
        $user->save();

        $this->dispatch('showToast', message: 'Visibility updated', type: 'success');
    }
};

?>

<div class="space-y-5 py-5">
    <x-page-header title="Users Management"
        subtitle="Overview of all registered users, their profiles, KYC status, and activity."
        back-url="{{ route('admin.dashboard') }}">
    </x-page-header>

    <x-table.wrapper :items="$this->users">

        {{-- SEARCH --}}
        <x-slot name="search">
            <div class="flex items-center justify-between gap-4">
                <div class="w-full md:w-1/3">
                    <div class="relative">
                        <x-flux::icon.magnifying-glass
                            class="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400 dark:text-gray-500" />
                        <input type="text" wire:model.live="search" placeholder="Search by name or email"
                            class="w-full pl-10 pr-4 py-2 text-sm rounded-md bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-sky-600 focus:border-sky-600 transition" />
                    </div>
                </div>
            </div>
        </x-slot>

        {{-- HEADERS --}}
        <x-slot name="head">
            <x-table.head label="User" />
            <x-table.head label="Balance" />
            <x-table.head label="KYC" />
            <x-table.head label="Profile Complete" />
            <x-table.head label="Country" />
            <x-table.head label="Joined" field="created_at" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Action" />
        </x-slot>

        {{-- BODY (DESKTOP) --}}
        <x-slot name="body">
            @forelse ($this->users as $user)
                @php
                    $profile = $user->profile;
                    $primaryAccount = $user->accounts()->where('is_primary', true)->first();
                    $profilePhoto = $profile?->profile_photo
                        ? Storage::disk('public')->url($profile->profile_photo)
                        : 'https://via.placeholder.com/40';
                    $currencySymbol = $profile ? \App\Support\Currency::symbol($profile->currency) : '$';
                @endphp

                <x-table.row wire:key="row-{{ $user->id }}"
                    class="hover:shadow-lg transition-shadow duration-200 rounded-lg mb-2">

                    {{-- User with photo + name + email --}}
                    <x-table.cell>
                        <a href="{{ route('admin.users.show', $user) }}" class="flex items-center gap-2">
                            @if ($profilePhoto && $profilePhoto !== 'https://via.placeholder.com/40')
                                <img src="{{ $profilePhoto }}" alt="User Avatar" class="w-6 h-6 rounded-full">
                            @else
                                <span
                                    class="w-6 h-6 flex items-center justify-center rounded-full bg-gray-300 text-gray-700 font-medium">
                                    {{ $user->initials() ?: '?' }}
                                </span>
                            @endif
                            <span>{{ $user->name ?? '-' }}</span>
                        </a>
                    </x-table.cell>


                    {{-- Balance --}}
                    <x-table.cell>
                        {{ $primaryAccount ? $currencySymbol : '-' }}
                        {{ $primaryAccount ? number_format($primaryAccount->available_balance, 2) : '-' }}
                    </x-table.cell>

                    {{-- KYC --}}
                    <x-table.cell>
                        @if ($user->kycStatus === \App\Models\Kyc::STATUS_VERIFIED)
                            <span class="px-2 py-1 rounded-full bg-green-500 text-white text-xs">Verified</span>
                        @elseif ($user->kycStatus === \App\Models\Kyc::STATUS_REJECTED)
                            <span class="px-2 py-1 rounded-full bg-red-500 text-white text-xs">Rejected</span>
                        @else
                            <span class="px-2 py-1 rounded-full bg-yellow-400 text-black text-xs">Pending</span>
                        @endif
                    </x-table.cell>


                    {{-- Profile Complete --}}
                    <x-table.cell>
                        @if ($user->isProfileComplete())
                            <span class="px-2 py-1 rounded-full bg-green-500 text-white text-xs">Complete</span>
                        @else
                            <span class="px-2 py-1 rounded-full bg-red-500 text-white text-xs">Incomplete</span>
                        @endif
                    </x-table.cell>

                    {{-- Country --}}
                    <x-table.cell>
                        {{ $profile?->country ?? '-' }}
                    </x-table.cell>

                    {{-- Joined --}}
                    <x-table.cell>{{ $user->created_at->diffForHumans() }}</x-table.cell>

                    {{-- Action --}}
                    <x-table.cell>
                        <a href="{{ route('admin.users.show', $user) }}"
                            class="inline-flex items-center px-3 py-1.5 text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-1 transition">
                            Edit
                        </a>

                        <!-- Delete Button -->
                        <button x-data
                            @click.prevent="if (confirm('Are you sure you want to delete this order?')) { $wire.delete({{ $user->id }}) }"
                            class="inline-flex items-center px-3 py-1.5 text-sm font-medium rounded-md text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-1 transition">
                            Delete
                        </button>

                        @can('updateVisibility', $user)
                            <x-select :options="[
                                0 => 'Normal User',
                                1 => 'Admin Only',
                                2 => 'Super Admin Only',
                            ]" wire:model="visibility.{{ $user->id }}"
                                wire:change="updateVisibility({{ $user->id }}, $event.target.value)"
                                placeholder="Visibility Level" size="sm" />
                        @endcan
                    </x-table.cell>

                </x-table.row>
            @empty
                <tr>
                    <td colspan="8" class="px-6 py-10 text-center text-gray-400 dark:text-gray-500">
                        No users found
                    </td>
                </tr>
            @endforelse
        </x-slot>
        <x-slot name="mobile">
            @forelse ($this->users as $user)
                @php
                    $profile = $user->profile;
                    $primaryAccount = $user->accounts()->where('is_primary', true)->first();
                    $profilePhoto = $profile?->profile_photo
                        ? Storage::disk('public')->url($profile->profile_photo)
                        : 'https://via.placeholder.com/40';
                    $currencySymbol = $profile ? \App\Support\Currency::symbol($profile->currency) : '$';
                @endphp

                <x-table.mobile-card
                    class="bg-white dark:bg-gray-800 rounded-xl shadow-md p-4 mb-4 hover:shadow-xl transition duration-200">

                    {{-- User + Photo + KYC --}}
                    <div class="flex items-center gap-3 mb-2">
                        <a href="{{ route('admin.users.show', $user) }}" class="flex items-center gap-2">
                            @if ($profilePhoto && $profilePhoto !== 'https://via.placeholder.com/40')
                                <img src="{{ $profilePhoto }}" alt="User Avatar" class="w-6 h-6 rounded-full">
                            @else
                                <span
                                    class="w-8 h-8 aspect-square flex items-center justify-center rounded-full bg-gray-300 text-gray-700 font-semibold uppercase">
                                    {{ $user->initials() ?: '?' }}
                                </span>
                            @endif
                            <div class="flex flex-col">
                                <span class="font-semibold text-gray-900 dark:text-gray-200">{{ $user->name }}</span>
                                <span
                                    class="text-gray-500 dark:text-gray-400 truncate text-sm">{{ $user->email }}</span>
                            </div>
                        </a>

                        <span
                            class="ml-auto text-xs font-semibold px-2 py-1 rounded-full
        {{ $user->kycStatus === \App\Models\Kyc::STATUS_VERIFIED
            ? 'bg-green-500 text-white'
            : ($user->kycStatus === \App\Models\Kyc::STATUS_REJECTED
                ? 'bg-red-500 text-white'
                : 'bg-yellow-400 text-black') }}">
                            {{ $user->kycStatus === \App\Models\Kyc::STATUS_VERIFIED
                                ? 'Verified'
                                : ($user->kycStatus === \App\Models\Kyc::STATUS_REJECTED
                                    ? 'Rejected'
                                    : 'Pending') }}
                        </span>

                    </div>

                    {{-- Balance + Currency + Profile + Country + Joined --}}
                    <div class="flex flex-col text-sm text-gray-700 dark:text-gray-300 space-y-1">
                        <div class="flex justify-between">
                            <span>Balance:</span>
                            <span>{{ $primaryAccount ? $currencySymbol : '-' }}{{ $primaryAccount ? number_format($primaryAccount->available_balance, 2) : '-' }}</span>
                        </div>

                        <div class="flex justify-between">
                            <span>Profile:</span>
                            <span>{{ $user->isProfileComplete() ? 'Complete' : 'Incomplete' }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span>Country:</span>
                            <span>{{ $profile?->country ?? '-' }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span>Joined:</span>
                            <span>{{ $user->created_at->diffForHumans() }}</span>
                        </div>
                    </div>

                    {{-- Actions: View, Edit, Delete --}}
                    <div class="mt-3 flex flex-col sm:flex-row sm:space-x-2 space-y-2 sm:space-y-0">
                        {{-- View --}}


                        {{-- Edit --}}
                        <a href="{{ route('admin.users.show', $user) }}"
                            class="flex-1 inline-flex items-center justify-center px-4 py-2 text-sm font-medium rounded-lg text-white bg-blue-600 hover:bg-blue-700 transition">
                            Edit
                        </a>

                        {{-- Delete --}}
                        <button x-data
                            @click.prevent="if (confirm('Are you sure you want to delete this user?')) { $wire.delete({{ $user->id }}) }"
                            class="flex-1 inline-flex items-center justify-center px-4 py-2 text-sm font-medium rounded-lg text-white bg-red-600 hover:bg-red-700 transition">
                            Delete
                        </button>

                        @can('updateVisibility', $user)
                            <x-select :options="[
                                0 => 'Normal User',
                                1 => 'Admin Only',
                                2 => 'Super Admin Only',
                            ]" wire:model="visibility.{{ $user->id }}"
                                wire:change="updateVisibility({{ $user->id }}, $event.target.value)"
                                placeholder="Visibility Level" size="sm" />
                        @endcan
                    </div>

                </x-table.mobile-card>
            @empty
                <div class="px-6 py-6 text-center text-gray-400 dark:text-gray-500">
                    No users found
                </div>
            @endforelse
        </x-slot>


        {{-- FOOTER / PAGINATION --}}
        <x-slot name="footer">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between w-full gap-4 mt-6"
                x-data="{
                    open: false,
                    options: [5, 10, 15, 20, 25, 50],
                    selected: @js($perPage)
                }" x-init="$watch('selected', value => $wire.set('perPage', value));
                Livewire.on('perPageUpdated', event => selected = event.value);">

                <div class="relative text-sm w-fit">
                    <label class="block text-gray-700 dark:text-gray-300 mb-1 text-xs font-medium">Rows per page</label>
                    <button @click="open = !open"
                        class="w-36 bg-white dark:bg-gray-800 text-gray-800 dark:text-white border border-gray-300 dark:border-gray-600 rounded-md px-4 py-2 flex justify-between items-center shadow-sm hover:border-gray-400 dark:hover:border-gray-500 focus:outline-none focus:ring-2 focus:ring-sky-500"
                        type="button" aria-haspopup="listbox" aria-expanded="open">
                        <span x-text="selected"></span>
                        <svg class="w-4 h-4 ml-2 transform transition-transform duration-200"
                            :class="{ 'rotate-180': open }" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7" />
                        </svg>
                    </button>
                    <div x-show="open" @click.outside="open = false"
                        x-transition:enter="transition ease-out duration-100"
                        x-transition:enter-start="opacity-0 scale-95" x-transition:enter-end="opacity-100 scale-100"
                        x-transition:leave="transition ease-in duration-75"
                        x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-95"
                        class="absolute bottom-0 mt-2 w-full z-50 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md shadow-lg">
                        <ul class="py-1 max-h-60 overflow-auto text-sm text-gray-800 dark:text-white">
                            <template x-for="option in options" :key="option">
                                <li>
                                    <a href="#" @click.prevent="selected = option; open = false"
                                        class="block px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700"
                                        :class="{ 'bg-gray-100 dark:bg-gray-700 font-semibold': selected === option }"
                                        x-text="option"></a>
                                </li>
                            </template>
                        </ul>
                    </div>
                </div>

                <div class="w-full md:w-auto text-center">{{ $this->users->links() }}</div>
            </div>
        </x-slot>

    </x-table.wrapper>


</div>
