<?php

use Livewire\Component;
use Livewire\Attributes\On;

new class extends Component {
    public int $unreadCount = 0;
    public array $notifications = [];
    public $admin;

    public function mount()
    {
        $this->admin = auth()->guard('admin')->user();

        $this->refreshUnreadCount();
        $this->loadNotifications();
    }

    /**
     * Load the latest notifications for the admin.
     */
    public function loadNotifications(): void
    {
        $admin = auth()->guard('admin')->user();

        $this->notifications = $admin->notifications()->latest()->take(5)->get()->toArray();
    }

    /**
     * Refresh unread notification count.
     */
    public function refreshUnreadCount(): void
    {
        $this->unreadCount = auth()->guard('admin')->user()->unreadNotifications()->count();
    }

    /**
     * Mark a notification as read.
     */
    public function markAsRead(string $id): void
    {
        $note = auth()->guard('admin')->user()->unreadNotifications()->find($id);

        if ($note) {
            $note->markAsRead();
        }

        $this->refreshUnreadCount();
        $this->loadNotifications();
    }

    /**
     * Listen for new notifications sent.
     */
    #[On('notification-sent')]
    public function handleNewNotification(): void
    {
        $this->refreshUnreadCount();
        $this->loadNotifications();
    }

    /**
     * Optional: listen for notifications read event.
     */
    #[On('notifications-read')]
    public function handleNotificationsRead(): void
    {
        $this->refreshUnreadCount();
        $this->loadNotifications();
    }
};
?>


<div
    class="px-6 py-4 bg-white dark:bg-zinc-900 shadow-sm border-b border-gray-200 dark:border-zinc-700 flex items-center justify-between">
    <!-- Mobile hamburger -->
    <div wire:ignore class="flex items-center gap-2">
        <div class="lg:hidden z-50">
            <button @click="opensidebar = !opensidebar"
                class="relative w-8 h-8 flex flex-col justify-center items-center gap-1.5 p-2 bg-white dark:bg-gray-800 text-rose-600 dark:text-white rounded-md shadow-md focus:outline-none focus:ring-2 focus:ring-rose-500">

                @foreach (['', '', ''] as $i)
                    <span
                        :class="opensidebar
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                ? ['first:rotate-45 absolute', 'second:opacity-0', 'third:-rotate-45 absolute'][$loop->index] 
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                : ''"
                        class="block w-6 h-0.5 bg-current transition-transform duration-500 ease-[cubic-bezier(0.68,-0.55,0.27,1.55)]
                           transition-opacity duration-500 scale-80"></span>
                @endforeach
            </button>
        </div>
    </div>

    <!-- Right: Notifications & Profile -->
    <div class="flex items-center gap-4">
        <!-- Dark Mode Toggle with animation -->
        <button @click="darkMode = !darkMode; localStorage.setItem('darkMode', darkMode)"
            class="relative p-3 rounded-2xl bg-linear-to-br from-gray-50 to-gray-100 
               dark:from-gray-800 dark:to-gray-700 text-gray-600 dark:text-gray-300 
               hover:from-primary-50 hover:to-primary-100 
               dark:hover:from-primary-900/30 dark:hover:to-primary-800/30 
               transition-all duration-300 shadow-lg hover:shadow-xl 
               hover:-translate-y-0.5 group">
            <x-flux::icon.sun variant="solid" x-show="!darkMode" x-cloak
                class="w-5 h-5 group-hover:rotate-180 transition-transform duration-500" />

            <x-flux::icon.moon variant="solid" x-cloak x-show="darkMode"
                class="w-5 h-5 group-hover:rotate-12 transition-transform duration-300" />
        </button>
        <!-- Notifications -->
        <div class="relative" x-data="{ openDropdown: null }">
            <button @click="openDropdown = (openDropdown === 'notifications' ? null : 'notifications')"
                class="relative p-2 rounded-full hover:bg-sky-100 dark:hover:bg-sky-900 focus:outline-none transition">
                <flux:icon.bell
                    class="w-6 h-6 {{ $unreadCount > 0 ? 'text-sky-500 animate-pulse' : 'text-gray-500 dark:text-gray-300' }}" />

                {{-- Badge --}}
                @if ($unreadCount > 0)
                    <span
                        class="absolute -top-1 -right-1 bg-sky-500 text-white rounded-full w-5 h-5 flex items-center justify-center text-[10px] font-semibold shadow">
                        {{ $unreadCount }}
                    </span>
                @endif
            </button>

            {{-- Dropdown --}}
            <div x-show="openDropdown === 'notifications'" x-collapse x-cloak
                class="absolute right-0 w-96 max-h-96 overflow-y-auto bg-white dark:bg-zinc-900 shadow-2xl rounded-xl border border-gray-200 dark:border-zinc-700 mt-2 z-50">

                <div class="py-2 px-3 border-b border-gray-200 dark:border-zinc-700">
                    <h4 class="text-sm font-semibold text-gray-700 dark:text-gray-200">Notifications</h4>
                </div>

                <div class="divide-y divide-gray-100 dark:divide-zinc-800">
                    @forelse($notifications as $notification)
                        @php
                            $link = $notification['data']['link'] ?? null;
                        @endphp

                        @if ($link)
                            <a href="{{ $link }}" class="block">
                        @endif

                        <div class="flex items-start gap-3 p-3 hover:bg-sky-50 dark:hover:bg-sky-900 rounded-lg transition cursor-pointer"
                            @if (!$link) wire:click="markAsRead('{{ $notification['id'] }}')" @endif>

                            <div class="shrink-0">
                                <flux:icon.bell-alert class="w-5 h-5 text-sky-500 mt-1" />
                            </div>

                            <div class="flex-1 min-w-0">
                                <p class="text-sm font-medium text-gray-900 dark:text-gray-100">
                                    {{ $notification['data']['title'] ?? 'Notification' }}
                                </p>

                                <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    {{ $notification['data']['message'] ?? '' }}
                                </p>

                                <span class="text-[10px] text-gray-400 dark:text-gray-500 mt-1 block">
                                    {{ \Illuminate\Support\Carbon::parse($notification['created_at'])->diffForHumans() }}
                                </span>
                            </div>

                            @if (!$notification['read_at'])
                                <span class="ml-2 shrink-0 w-2 h-2 bg-sky-500 rounded-full mt-1"></span>
                            @endif
                        </div>

                        @if ($link)
                            </a>
                        @endif
                    @empty
                        <p class="text-center text-gray-400 dark:text-gray-500 py-6 text-sm">
                            No new notifications
                        </p>
                    @endforelse
                </div>

                <div class="text-center py-2">
                    <a href="{{ route('user.transactions') }}"
                        class="text-sky-600 hover:text-sky-700 dark:text-sky-400 dark:hover:text-sky-300 text-sm font-medium transition">
                        View All
                    </a>
                </div>
            </div>
        </div>


        <!-- Profile -->
        <div
            class="flex items-center gap-3 px-3 py-2 rounded-full bg-white/60 dark:bg-gray-900/60
            ring-1 ring-gray-200 dark:ring-gray-700 shadow-sm">

            <!-- Name -->
            <div class="text-right leading-tight">
                <p class="text-sm font-semibold text-gray-900 dark:text-gray-100">
                    {{ $admin->name ?? 'Admin' }}
                </p>
                <p class="text-[11px] text-gray-500 dark:text-gray-400">
                    Administrator
                </p>
            </div>

            <!-- Avatar -->
            @if ($admin->avatar)
                <img src="{{ Storage::disk('public')->url($admin->avatar) }}"
                    class="w-12 h-12 rounded-full object-cover shadow" />
            @else
                <div
                    class="w-12 h-12 rounded-full bg-gray-200 dark:bg-gray-700 flex items-center justify-center text-gray-500 dark:text-gray-400 font-semibold text-lg">
                    {{ strtoupper(substr($admin->name, 0, 1)) }}
                </div>
            @endif

        </div>

    </div>
</div>
