<?php

namespace Database\Seeders;

use App\Models\CryptoWallet;
use App\Models\PaymentMethod;
use Illuminate\Database\Seeder;

class PaymentMethodSeeder extends Seeder
{
    public function run()
    {
        // Clear old data
        CryptoWallet::query()->delete();
        PaymentMethod::query()->delete();

        $methods = [

            /*
            |--------------------------------------------------------------------------
            | Deposit Methods
            |--------------------------------------------------------------------------
            */

            [
                'type' => 'deposit',
                'name' => 'Crypto',
                'instructions' => [
                    'deposit' => [
                        'Send the exact amount to the selected wallet.',
                        'Confirm the transaction hash after sending.'
                    ]
                ],
                'details' => [
                    'deposit' => [
                        ['field1' => 'Supported Coins', 'field2' => 'BTC, ETH, USDT, BNB, LTC']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'tx_hash',
                            'label' => 'Transaction Hash',
                            'type' => 'text',
                            'required' => true
                        ]
                    ]
                ],
                'fee_percent' => 1.5,
                'is_active' => true
            ],

            [
                'type' => 'deposit',
                'name' => 'Bank Transfer',
                'instructions' => [
                    'deposit' => [
                        'Transfer the amount to our bank account.',
                        'Include your user ID in the transfer reference.'
                    ]
                ],
                'details' => [
                    'deposit' => [
                        ['field1' => 'Account Name', 'field2' => 'Novabank Ltd'],
                        ['field1' => 'Account Number', 'field2' => '9876543210'],
                        ['field1' => 'Bank', 'field2' => 'Chase Bank, Bank of America, Wells Fargo']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'deposit_slip',
                            'label' => 'Deposit Slip',
                            'type' => 'file',
                            'required' => true
                        ]
                    ]
                ],
                'fee_percent' => 0.5,
                'is_active' => true
            ],

            [
                'type' => 'deposit',
                'name' => 'PayPal',
                'instructions' => [
                    'deposit' => [
                        'Send the payment to paypal@novabank.com.',
                        'Use your registered email as reference.'
                    ]
                ],
                'details' => [
                    'deposit' => [
                        ['field1' => 'PayPal Email', 'field2' => 'paypal@novabank.com']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'paypal_txn_id',
                            'label' => 'PayPal Transaction ID',
                            'type' => 'text',
                            'required' => true
                        ]
                    ]
                ],
                'fee_percent' => 2.0,
                'is_active' => true
            ],

            /*
            |--------------------------------------------------------------------------
            | Withdrawal Methods
            |--------------------------------------------------------------------------
            */

            [
                'type' => 'withdrawal',
                'name' => 'Crypto',
                'instructions' => [
                    'withdrawal' => [
                        'Select your cryptocurrency.',
                        'Choose the correct network.',
                        'Enter the destination wallet address carefully.',
                        'Include destination tag/memo if required by the coin.',
                        'Withdrawals are processed after confirmation.'
                    ]
                ],
                'details' => [
                    'withdrawal' => [
                        ['field1' => 'Network Fee', 'field2' => 'Varies by blockchain']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'coin',
                            'label' => 'Select Coin',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'Bitcoin (BTC)', 'value' => 'btc'],
                                ['label' => 'Ethereum (ETH)', 'value' => 'eth'],
                                ['label' => 'Tether (USDT)', 'value' => 'usdt'],
                                ['label' => 'Binance Coin (BNB)', 'value' => 'bnb'],
                                ['label' => 'Litecoin (LTC)', 'value' => 'ltc'],
                            ]
                        ],
                        [
                            'name' => 'network',
                            'label' => 'Network',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'ERC20', 'value' => 'erc20'],
                                ['label' => 'TRC20', 'value' => 'trc20'],
                                ['label' => 'BEP20', 'value' => 'bep20'],
                                ['label' => 'Bitcoin (BTC)', 'value' => 'btc'],
                                ['label' => 'Litecoin (LTC)', 'value' => 'ltc'],
                            ]
                        ],
                        [
                            'name' => 'wallet_address',
                            'label' => 'Destination Wallet Address',
                            'type' => 'text',
                            'required' => true
                        ],
                        [
                            'name' => 'destination_tag',
                            'label' => 'Destination Tag / Memo (if applicable)',
                            'type' => 'text',
                            'required' => false
                        ]
                    ]
                ],
                'fee_percent' => 1.2,
                'is_active' => true
            ],

            [
                'type' => 'withdrawal',
                'name' => 'Bank Transfer',
                'instructions' => [
                    'withdrawal' => [
                        'Enter your bank details correctly.',
                        'Withdrawals may take 1–3 business days.'
                    ]
                ],
                'details' => [
                    'withdrawal' => [
                        ['field1' => 'Processing Time', 'field2' => '1–3 Business Days']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'bank_name',
                            'label' => 'Bank Name',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'Chase', 'value' => 'chase'],
                                ['label' => 'Bank of America', 'value' => 'boa'],
                                ['label' => 'Wells Fargo', 'value' => 'wells_fargo'],
                                ['label' => 'Citibank', 'value' => 'citibank'],
                            ]
                        ],
                        [
                            'name' => 'account_name',
                            'label' => 'Account Holder Name',
                            'type' => 'text',
                            'required' => true
                        ],
                        [
                            'name' => 'account_number',
                            'label' => 'Account Number',
                            'type' => 'number',
                            'required' => true
                        ]
                    ]
                ],
                'fee_percent' => 0.8,
                'is_active' => true
            ],

            [
                'type' => 'withdrawal',
                'name' => 'PayPal',
                'instructions' => [
                    'withdrawal' => [
                        'Withdrawals are sent to your PayPal email.',
                        'Ensure the email is active and correct.'
                    ]
                ],
                'details' => [
                    'withdrawal' => [
                        ['field1' => 'Processing Time', 'field2' => 'Within 24 hours']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'paypal_email',
                            'label' => 'PayPal Email',
                            'type' => 'text',
                            'required' => true
                        ]
                    ]
                ],
                'fee_percent' => 2.5,
                'is_active' => true
            ],

            /*
|--------------------------------------------------------------------------
| Local Transfer Methods
|--------------------------------------------------------------------------
*/

            [
                'type' => 'local_transfer',
                'name' => 'ACH / Local Bank',
                'instructions' => [
                    'local_transfer' => [
                        'Transfer funds from your local bank account.',
                        'Include your user ID in the reference for tracking.'
                    ]
                ],
                'details' => [
                    'local_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => '1–2 business days'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.5%']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        ['name' => 'account_number', 'label' => 'Account Number', 'type' => 'number', 'required' => true],
                        ['name' => 'routing_number', 'label' => 'Routing Number', 'type' => 'text', 'required' => true],
                        ['name' => 'account_name', 'label' => 'Account Holder Name', 'type' => 'text', 'required' => true],
                    ]
                ],
                'fee_percent' => 0.5,
                'is_active' => true
            ],

            [
                'type' => 'local_transfer',
                'name' => 'Zelle',
                'instructions' => [
                    'local_transfer' => [
                        'Send funds directly via Zelle using your registered email or phone.',
                        'Transfers are usually instant.'
                    ]
                ],
                'details' => [
                    'local_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Instant'],
                        ['field1' => 'Transfer Fee', 'field2' => 'Free']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'zelle_email', 'label' => 'Zelle Email / Phone', 'type' => 'text', 'required' => true],
                    ]
                ],
                'fee_percent' => 0,
                'is_active' => true
            ],

            [
                'type' => 'local_transfer',
                'name' => 'Venmo',
                'instructions' => [
                    'local_transfer' => [
                        'Send money via Venmo to our registered account.',
                        'Include your user ID in the note.'
                    ]
                ],
                'details' => [
                    'local_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Instant'],
                        ['field1' => 'Transfer Fee', 'field2' => '1% for credit card, free for bank transfer']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'venmo_username', 'label' => 'Venmo Username', 'type' => 'text', 'required' => true],
                    ]
                ],
                'fee_percent' => 0.5,
                'is_active' => true
            ],

            [
                'type' => 'local_transfer',
                'name' => 'Faster Payments (UK)',
                'instructions' => [
                    'local_transfer' => [
                        'Send money from your UK bank account using Faster Payments.',
                        'Include your reference ID in the payment description.'
                    ]
                ],
                'details' => [
                    'local_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Instant – Same Day'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.2%']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        ['name' => 'account_number', 'label' => 'Account Number', 'type' => 'text', 'required' => true],
                        ['name' => 'sort_code', 'label' => 'Sort Code', 'type' => 'text', 'required' => true],
                        ['name' => 'account_name', 'label' => 'Account Holder Name', 'type' => 'text', 'required' => true],
                    ]
                ],
                'fee_percent' => 0.2,
                'is_active' => true
            ],

            /*
            |--------------------------------------------------------------------------
            | International Transfer Methods
            |--------------------------------------------------------------------------
            */

            // SWIFT / Wire Transfer
            [
                'type' => 'international_transfer',
                'name' => 'SWIFT / Wire Transfer',
                'instructions' => [
                    'international_transfer' => [
                        'Enter beneficiary details carefully.',
                        'Include the SWIFT/BIC code.',
                        'Transfers may take 1–5 business days depending on the bank.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Transfer Fee', 'field2' => 'Varies by bank']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'beneficiary_name', 'label' => 'Beneficiary Name', 'type' => 'text', 'required' => true],
                        ['name' => 'iban', 'label' => 'IBAN', 'type' => 'text', 'required' => true],
                        ['name' => 'swift_code', 'label' => 'SWIFT / BIC', 'type' => 'text', 'required' => true],
                        ['name' => 'bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        ['name' => 'bank_address', 'label' => 'Bank Address', 'type' => 'text', 'required' => false],
                        ['name' => 'country', 'label' => 'Country', 'type' => 'text', 'required' => true],
                        ['name' => 'reference', 'label' => 'Reference / Memo', 'type' => 'textarea', 'required' => false],
                        ['name' => 'routing_number', 'label' => 'Routing Number', 'type' => 'text', 'required' => false],
                    ]
                ],
                'fee_percent' => 1.5,
                'is_active' => true
            ],

            // PayPal International
            [
                'type' => 'international_transfer',
                'name' => 'PayPal International',
                'instructions' => [
                    'international_transfer' => [
                        'Send the payment to the recipient PayPal email.',
                        'Ensure the email is correct to avoid loss of funds.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Within 24 hours'],
                        ['field1' => 'Transfer Fee', 'field2' => '1–3% depending on currency']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'paypal_email', 'label' => 'Recipient PayPal Email', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'USD', 'value' => 'usd'],
                                ['label' => 'EUR', 'value' => 'eur'],
                                ['label' => 'GBP', 'value' => 'gbp'],
                                ['label' => 'JPY', 'value' => 'jpy'],
                                ['label' => 'AUD', 'value' => 'aud'],
                            ]
                        ],
                        ['name' => 'transaction_id', 'label' => 'Transaction ID', 'type' => 'text', 'required' => true],
                        ['name' => 'note', 'label' => 'Note / Reference', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 2.0,
                'is_active' => true
            ],

            // Wise / Multi-currency Transfer
            [
                'type' => 'international_transfer',
                'name' => 'Wise / Multi-currency Transfer',
                'instructions' => [
                    'international_transfer' => [
                        'Provide recipient bank details accurately.',
                        'Select the currency for the transfer.',
                        'Wise will convert and send the funds securely.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Transfer Limit', 'field2' => '$1 - $50,000 per transfer'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.5% - 1% per transaction']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'recipient_name', 'label' => 'Recipient Name', 'type' => 'text', 'required' => true],
                        ['name' => 'recipient_iban', 'label' => 'IBAN', 'type' => 'text', 'required' => true],
                        ['name' => 'recipient_bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'USD', 'value' => 'usd'],
                                ['label' => 'EUR', 'value' => 'eur'],
                                ['label' => 'GBP', 'value' => 'gbp'],
                                ['label' => 'JPY', 'value' => 'jpy'],
                                ['label' => 'AUD', 'value' => 'aud'],
                            ]
                        ],
                        ['name' => 'reference', 'label' => 'Reference / Note', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 1.0,
                'is_active' => true
            ],

            // Revolut Transfer
            [
                'type' => 'international_transfer',
                'name' => 'Revolut Transfer',
                'instructions' => [
                    'international_transfer' => [
                        'Provide the beneficiary IBAN and bank details.',
                        'Select currency and amount carefully.',
                        'Transfers are usually instant or within 1 business day.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Transfer Fee', 'field2' => '0.5% - 1% depending on currency'],
                        ['field1' => 'Processing Time', 'field2' => 'Instant – 1 business day']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'beneficiary_name', 'label' => 'Beneficiary Name', 'type' => 'text', 'required' => true],
                        ['name' => 'iban', 'label' => 'IBAN', 'type' => 'text', 'required' => true],
                        ['name' => 'bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'USD', 'value' => 'usd'],
                                ['label' => 'EUR', 'value' => 'eur'],
                                ['label' => 'GBP', 'value' => 'gbp'],
                                ['label' => 'JPY', 'value' => 'jpy'],
                                ['label' => 'AUD', 'value' => 'aud'],
                            ]
                        ],
                        ['name' => 'note', 'label' => 'Note / Reference', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 0.8,
                'is_active' => true
            ],

            // Crypto International Transfer
            [
                'type' => 'international_transfer',
                'name' => 'Crypto International',
                'instructions' => [
                    'international_transfer' => [
                        'Select the cryptocurrency and network carefully.',
                        'Send to the correct wallet address.',
                        'Confirm transaction hash for security.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Supported Coins', 'field2' => 'BTC, ETH, USDT, BNB, LTC'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.1% - 0.5% depending on blockchain']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        [
                            'name' => 'coin',
                            'label' => 'Select Coin',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'Bitcoin (BTC)', 'value' => 'btc'],
                                ['label' => 'Ethereum (ETH)', 'value' => 'eth'],
                                ['label' => 'Tether (USDT)', 'value' => 'usdt'],
                                ['label' => 'Binance Coin (BNB)', 'value' => 'bnb'],
                                ['label' => 'Litecoin (LTC)', 'value' => 'ltc'],
                            ]
                        ],
                        [
                            'name' => 'network',
                            'label' => 'Network',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'ERC20', 'value' => 'erc20'],
                                ['label' => 'TRC20', 'value' => 'trc20'],
                                ['label' => 'BEP20', 'value' => 'bep20'],
                                ['label' => 'Bitcoin (BTC)', 'value' => 'btc'],
                                ['label' => 'Litecoin (LTC)', 'value' => 'ltc'],
                            ]
                        ],
                        ['name' => 'wallet_address', 'label' => 'Destination Wallet Address', 'type' => 'text', 'required' => true],
                        ['name' => 'transaction_hash', 'label' => 'Transaction Hash', 'type' => 'text', 'required' => true],
                        ['name' => 'note', 'label' => 'Note / Reference', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 1.2,
                'is_active' => true
            ],

            // Bank Draft
            [
                'type' => 'international_transfer',
                'name' => 'Bank Draft',
                'instructions' => [
                    'international_transfer' => [
                        'Provide beneficiary and bank details accurately.',
                        'Ensure the draft is issued in the correct currency.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Transfer Fee', 'field2' => '0.7% of the transfer amount'],
                        ['field1' => 'Processing Time', 'field2' => '2–7 business days']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'beneficiary_name', 'label' => 'Beneficiary Name', 'type' => 'text', 'required' => true],
                        ['name' => 'iban', 'label' => 'IBAN / Account Number', 'type' => 'text', 'required' => true],
                        ['name' => 'bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'USD', 'value' => 'usd'],
                                ['label' => 'EUR', 'value' => 'eur'],
                                ['label' => 'GBP', 'value' => 'gbp'],
                            ]
                        ],
                        ['name' => 'note', 'label' => 'Note / Reference', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 0.7,
                'is_active' => true
            ],

            // SEPA Transfer
            [
                'type' => 'international_transfer',
                'name' => 'SEPA Transfer',
                'instructions' => [
                    'international_transfer' => [
                        'Used for transfers within the EU.',
                        'Provide beneficiary IBAN and BIC/SWIFT code.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => '1–2 business days'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.2%']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'beneficiary_name', 'label' => 'Beneficiary Name', 'type' => 'text', 'required' => true],
                        ['name' => 'iban', 'label' => 'IBAN', 'type' => 'text', 'required' => true],
                        ['name' => 'bic', 'label' => 'BIC / SWIFT', 'type' => 'text', 'required' => true],
                        ['name' => 'bank_name', 'label' => 'Bank Name', 'type' => 'text', 'required' => true],
                        ['name' => 'note', 'label' => 'Reference / Note', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 0.2,
                'is_active' => true
            ],

            /*
|--------------------------------------------------------------------------
| Additional International Transfer Methods
|--------------------------------------------------------------------------
*/

            [
                'type' => 'international_transfer',
                'name' => 'Payoneer',
                'instructions' => [
                    'international_transfer' => [
                        'Send funds to the recipient’s Payoneer account.',
                        'Use the correct email or Payoneer account ID.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => '1–3 business days'],
                        ['field1' => 'Transfer Fee', 'field2' => '1–3% depending on currency']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'payoneer_email', 'label' => 'Recipient Payoneer Email', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'USD', 'value' => 'usd'],
                                ['label' => 'EUR', 'value' => 'eur'],
                                ['label' => 'GBP', 'value' => 'gbp'],
                            ]
                        ],
                        ['name' => 'note', 'label' => 'Note / Reference', 'type' => 'textarea', 'required' => false],
                    ]
                ],
                'fee_percent' => 1.5,
                'is_active' => true
            ],

            [
                'type' => 'international_transfer',
                'name' => 'Western Union',
                'instructions' => [
                    'international_transfer' => [
                        'Send cash via Western Union to the recipient.',
                        'Provide the MTCN and beneficiary details.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Within 24 hours for cash pickup'],
                        ['field1' => 'Transfer Fee', 'field2' => 'Varies by amount & destination']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'recipient_name', 'label' => 'Beneficiary Name', 'type' => 'text', 'required' => true],
                        ['name' => 'recipient_country', 'label' => 'Country', 'type' => 'text', 'required' => true],
                        ['name' => 'mtcn', 'label' => 'MTCN', 'type' => 'text', 'required' => true],
                    ]
                ],
                'fee_percent' => 2.0,
                'is_active' => true
            ],

            [
                'type' => 'international_transfer',
                'name' => 'Alipay (China)',
                'instructions' => [
                    'international_transfer' => [
                        'Send funds to the recipient via Alipay.',
                        'Ensure the recipient’s account is verified.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Within 24 hours'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.5% - 1%']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'alipay_id', 'label' => 'Recipient Alipay ID', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'CNY', 'value' => 'cny'],
                                ['label' => 'USD', 'value' => 'usd'],
                            ]
                        ],
                    ]
                ],
                'fee_percent' => 0.8,
                'is_active' => true
            ],

            [
                'type' => 'international_transfer',
                'name' => 'WeChat Pay (China)',
                'instructions' => [
                    'international_transfer' => [
                        'Send funds via WeChat Pay to the recipient’s account.',
                        'Ensure the account is verified and active.'
                    ]
                ],
                'details' => [
                    'international_transfer' => [
                        ['field1' => 'Processing Time', 'field2' => 'Instant – 1 business day'],
                        ['field1' => 'Transfer Fee', 'field2' => '0.5%']
                    ]
                ],
                'fields' => [
                    'fields' => [
                        ['name' => 'wechat_id', 'label' => 'Recipient WeChat ID', 'type' => 'text', 'required' => true],
                        [
                            'name' => 'currency',
                            'label' => 'Currency',
                            'type' => 'select',
                            'required' => true,
                            'options' => [
                                ['label' => 'CNY', 'value' => 'cny'],
                                ['label' => 'USD', 'value' => 'usd'],
                            ]
                        ],
                    ]
                ],
                'fee_percent' => 0.5,
                'is_active' => true
            ],

        ];

        // Insert / update payment methods
        foreach ($methods as $m) {
            PaymentMethod::updateOrCreate(
                ['type' => $m['type'], 'name' => $m['name']],
                [
                    'logo' => $m['logo'] ?? null,
                    'instructions' => $m['instructions'] ?? [],
                    'details' => $m['details'] ?? [],
                    'fields' => $m['fields'] ?? [],
                    'fee_percent' => $m['fee_percent'] ?? 0,
                    'is_active' => $m['is_active'] ?? true,
                ]
            );
        }

        // Crypto wallets for Crypto Deposit
        $crypto = PaymentMethod::where('type', 'deposit')->where('name', 'Crypto')->first();
        $wallets = [
            ['code' => 'BTC', 'name' => 'Bitcoin', 'address' => '1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa'],
            ['code' => 'ETH', 'name' => 'Ethereum', 'address' => '0x742d35Cc6634C0532925a3b844Bc454e4438f44e'],
            ['code' => 'USDT', 'name' => 'Tether', 'address' => 'THTE1234567890ABCDEFGHIJKLMN'],
            ['code' => 'BNB', 'name' => 'Binance Coin', 'address' => 'bnb1qxyzabcd1234567890abcdefg'],
            ['code' => 'LTC', 'name' => 'Litecoin', 'address' => 'Lc1234567890abcdefghijklmno'],
        ];
        foreach ($wallets as $w) {
            CryptoWallet::updateOrCreate([
                'payment_method_id' => $crypto->id,
                'code' => $w['code'],
                'name' => $w['name'],
                'address' => $w['address'],
                'is_active' => true,
            ]);
        }
    }
}
