<?php

namespace App\Services;

use App\Models\Account;
use App\Models\Transfer;
use App\Models\Transaction;
use App\Models\WalletReserve;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Notifications\TransactionNotification;

class TransferHandler
{
      /**
       * Approve a transfer.
       */
      public function approveTransfer(Transfer $transfer): void
      {
            DB::transaction(function () use ($transfer) {
                  $receiverTransaction = null; // initialize for scope

                  $transfer = Transfer::where('id', $transfer->id)
                        ->lockForUpdate()
                        ->firstOrFail();

                  if ($transfer->status !== Transfer::STATUS_PENDING) {
                        Log::warning("Transfer not pending", ['transfer_id' => $transfer->id]);
                        return;
                  }

                  $sender = $transfer->account()->lockForUpdate()->firstOrFail();
                  $senderTransaction = $this->getTransaction($transfer->id, 'transfer_out');

                  if ($transfer->type === Transfer::TYPE_INTERNATIONAL) {
                        $this->approveInternational($transfer, $sender, $senderTransaction);
                  } else {
                        $receiver = $this->getReceiverAccount($transfer);
                        $receiverTransaction = $this->approveLocal($transfer, $sender, $senderTransaction, $receiver);
                  }

                  $transfer->update(['status' => Transfer::STATUS_COMPLETED]);

                  DB::afterCommit(function () use ($senderTransaction, $receiverTransaction) {
                        $this->notifyTransaction($senderTransaction);

                        // Only notify receiver if it's a local transfer
                        if ($receiverTransaction) {
                              $this->notifyTransaction($receiverTransaction);
                        }
                  });
            });
      }


      /**
       * Reject a transfer.
       */
      public function rejectTransfer(Transfer $transfer, string $reason = 'Transfer rejected'): void
      {
            DB::transaction(function () use ($transfer, $reason) {
                  $transfer = Transfer::where('id', $transfer->id)
                        ->lockForUpdate()
                        ->first();

                  if (!$transfer || $transfer->status !== Transfer::STATUS_PENDING)
                        return;

                  $senderTransaction = $this->getTransaction($transfer->id, 'transfer_out');

                  if ($senderTransaction) {
                        $senderTransaction->update([
                              'status' => Transaction::STATUS_REJECTED,
                              'description' => $reason,
                        ]);
                  }

                  $this->releaseWalletReserve($transfer->account_id);

                  $sender = $transfer->account()->lockForUpdate()->first();
                  if ($sender) {
                        $sender->balance = bcadd((string) $sender->balance, (string) $transfer->total, 2);
                        $sender->save();
                  }

                  $transfer->update([
                        'status' => Transfer::STATUS_FAILED,
                        'rejection_reason' => $reason,
                  ]);

                  DB::afterCommit(function () use ($senderTransaction) {
                        $this->notifyTransaction($senderTransaction);
                  });
            });
      }

      /**
       * Approve international transfer.
       */
      protected function approveInternational(Transfer $transfer, Account $sender, Transaction $senderTransaction): void
      {
            if ($sender->available_balance < $transfer->total) {
                  throw new \Exception("Insufficient balance for transfer ID {$transfer->id}");
            }

            $sender->balance = bcsub((string) $sender->balance, (string) $transfer->total, 2);
            $sender->save();

            $senderTransaction->update([
                  'status' => Transaction::STATUS_APPROVED,
                  'balance_after' => $sender->balance,
                  'description' => 'International transfer approved',
            ]);
      }

      /**
       * Approve local transfer.
       */
      protected function approveLocal(Transfer $transfer, Account $sender, Transaction $senderTransaction, Account $receiver): Transaction
      {
            if ($sender->available_balance < $transfer->total) {
                  throw new \Exception("Insufficient balance for local transfer ID {$transfer->id}");
            }

            // Debit sender
            $sender->balance = bcsub((string) $sender->balance, (string) $transfer->total, 2);
            $sender->save();
            $senderTransaction->update([
                  'status' => Transaction::STATUS_APPROVED,
                  'balance_after' => $sender->balance,
                  'description' => 'Local transfer approved',
            ]);

            // Credit receiver
            $receiver->balance = bcadd((string) $receiver->balance, (string) $transfer->amount, 2);
            $receiver->save();

            // Update or create receiver transaction
            $receiverTransaction = Transaction::where('transfer_id', $transfer->id)
                  ->where('type', 'transfer_in')
                  ->lockForUpdate()
                  ->first();

            if (!$receiverTransaction) {
                  $receiverTransaction = Transaction::create([
                        'account_id' => $receiver->id,
                        'transfer_id' => $transfer->id,
                        'type' => 'transfer_in',
                        'amount' => $transfer->amount,
                        'fee' => 0,
                        'total' => $transfer->amount,
                        'status' => Transaction::STATUS_APPROVED,
                        'balance_after' => $receiver->balance,
                        'description' => 'Received local transfer',
                  ]);
            } else {
                  $receiverTransaction->update([
                        'status' => Transaction::STATUS_APPROVED,
                        'balance_after' => $receiver->balance,
                        'description' => 'Received local transfer',
                  ]);
            }

            return $receiverTransaction;
      }

      /**
       * Get transaction with lock.
       */
      protected function getTransaction(int $transferId, string $type): Transaction
      {
            $transaction = Transaction::where('transfer_id', $transferId)
                  ->where('type', $type)
                  ->lockForUpdate()
                  ->first();

            if (!$transaction) {
                  throw new \Exception("Transaction not found for transfer ID {$transferId}");
            }

            return $transaction;
      }

      /**
       * Get receiver account for local transfer.
       */
      protected function getReceiverAccount(Transfer $transfer): Account
      {
            $receiverAccountNumber = $transfer->meta['account_number'] ?? null;
            if (!$receiverAccountNumber) {
                  throw new \Exception("Receiver account number missing for local transfer ID {$transfer->id}");
            }

            $receiver = Account::where('account_number', $receiverAccountNumber)
                  ->lockForUpdate()
                  ->first();

            if (!$receiver) {
                  throw new \Exception("Receiver account not found for local transfer ID {$transfer->id}");
            }

            return $receiver;
      }

      /**
       * Release wallet reserve.
       */
      protected function releaseWalletReserve(int $accountId): void
      {
            $reserve = WalletReserve::where('account_id', $accountId)
                  ->where('action_type', 'transfer')
                  ->lockForUpdate()
                  ->first();

            if ($reserve && $reserve->status === 'pending') {
                  $reserve->update([
                        'status' => WalletReserve::STATUS_APPROVED,
                        'approved_at' => now(),
                  ]);
            }
      }

      /**
       * Notify transaction after commit.
       */
      protected function notifyTransaction(Transaction $transaction): void
      {
            try {
                  $transaction->account->profile?->user?->notify(
                        new TransactionNotification($transaction)
                  );
            } catch (\Throwable $e) {
                  Log::error('Error sending transaction notification', ['exception' => $e]);
            }
      }
}
