<?php

namespace App\Services;

use App\Models\Grant;
use App\Models\GrantLog;
use Illuminate\Support\Facades\DB;
use App\Notifications\GrantStatusNotification;
use Exception;

class AdminGrantService
{
      /**
       * Approve a grant and optionally disburse funds
       */
      public function approveGrant(Grant $grant, ?string $note = null, bool $disburse = false): Grant
      {
            $this->ensurePending($grant, 'approve');

            return DB::transaction(function () use ($grant, $note, $disburse) {
                  $grant->update(['status' => 'approved']);

                  $disbursed = false;
                  if ($disburse && bccomp($grant->amount, '0', 2) === 1) {
                        $this->disburseGrantFunds($grant);
                        $disbursed = true;
                  }

                  $this->logGrantAction($grant, 'approved', [
                        'note' => $note,
                        'disbursed' => $disbursed,
                  ]);

                  $this->notifyUser($grant, 'approved', $note);

                  return $grant;
            });
      }

      /**
       * Reject a grant
       */
      public function rejectGrant(Grant $grant, string $reason): Grant
      {
            $this->ensurePending($grant, 'reject');

            return DB::transaction(function () use ($grant, $reason) {
                  $grant->update(['status' => 'rejected']);

                  $this->logGrantAction($grant, 'rejected', ['reason' => $reason]);

                  $this->notifyUser($grant, 'rejected', $reason);

                  return $grant;
            });
      }

      /**
       * Put a grant on hold
       */
      public function holdGrant(Grant $grant, ?string $reason = null): Grant
      {
            $this->ensurePending($grant, 'hold');

            return DB::transaction(function () use ($grant, $reason) {
                  $grant->update(['status' => 'on_hold']);

                  $this->logGrantAction($grant, 'on_hold', ['reason' => $reason]);

                  $this->notifyUser($grant, 'on_hold', $reason);

                  return $grant;
            });
      }

      /**
       * Bulk action handler
       */
      protected function bulkAction(array $grantIds, string $action, $param = null): int
      {
            $grants = Grant::whereIn('id', $grantIds)->where('status', 'pending')->get();
            $count = 0;

            DB::transaction(function () use ($grants, $action, $param, &$count) {
                  foreach ($grants as $grant) {
                        if ($action === 'approve') {
                              $disburse = is_array($param) && $param['disburse'] ?? false;
                              $note = is_array($param) ? $param['note'] ?? null : null;
                              $this->approveGrant($grant, $note, $disburse);
                        } elseif ($action === 'reject') {
                              $this->rejectGrant($grant, (string) $param);
                        } elseif ($action === 'hold') {
                              $this->holdGrant($grant, (string) $param);
                        }
                        $count++;
                  }
            });

            return $count;
      }

      public function bulkApprove(array $grantIds, ?string $note = null, bool $disburse = false): int
      {
            return $this->bulkAction($grantIds, 'approve', ['note' => $note, 'disburse' => $disburse]);
      }

      public function bulkReject(array $grantIds, string $reason): int
      {
            return $this->bulkAction($grantIds, 'reject', $reason);
      }

      public function bulkHold(array $grantIds, ?string $reason = null): int
      {
            return $this->bulkAction($grantIds, 'hold', $reason);
      }

      /**
       * Ensure grant is pending
       */
      protected function ensurePending(Grant $grant, string $action)
      {
            if ($grant->status !== 'pending') {
                  throw new Exception("Cannot {$action} grant. Only pending grants can be modified.");
            }
      }

      /**
       * Disburse funds using BCMath for precision
       */
      protected function disburseGrantFunds(Grant $grant): void
      {
            $user = $grant->user;

            // Try to get primary account, else fallback to the first account
            $account = $user->profile->accounts()->where('is_primary', true)->first()
                  ?? $user->profile->accounts()->first();

            if (!$account) {
                  throw new \Exception("No account found for user ID {$user->id}");
            }

            $currentBalance = bcadd((string) $account->balance, '0', 2);
            $newBalance = bcadd($currentBalance, (string) $grant->amount, 2);

            $account->balance = $newBalance;
            $account->save();

            $this->logGrantAction($grant, 'disbursed', [
                  'amount' => $grant->amount,
                  'balance_after' => $newBalance,
                  'account_id' => $account->id,
            ]);
      }


      /**
       * Log grant action
       */
      protected function logGrantAction(Grant $grant, string $action, array $data = []): void
      {
            // Find the original submission log by the grant owner
            $originalLog = GrantLog::where('grant_id', $grant->id)
                  ->where('user_id', $grant->user_id)
                  ->where('action', 'submitted')
                  ->first();

            if (!$originalLog) {
                  // If there is no submission log, just skip (do not create a new log)
                  return;
            }

            // Merge new data into the existing log
            $originalData = $originalLog->data ?? [];
            $originalLog->update([
                  'action' => $action,          // e.g., 'approved', 'disbursed', 'rejected'
                  'data' => array_merge($originalData, $data),
            ]);
      }
      /**
       * Delete a single grant (optionally reject before deleting)
       */
      public function deleteGrant(Grant $grant, ?string $reason = 'Deleted by admin'): void
      {
            DB::transaction(function () use ($grant, $reason) {
                  // Reject before deleting if still pending
                  if ($grant->status === 'pending') {
                        $this->rejectGrant($grant, $reason);
                  }

                  $grant->delete();

                  // Optionally, you could log a deletion action separately if needed
                  $this->logGrantAction($grant, 'deleted', ['reason' => $reason]);
            });
      }

      /**
       * Bulk delete grants
       */
      public function bulkDelete(array $grantIds, ?string $reason = 'Deleted by admin'): int
      {
            $grants = Grant::whereIn('id', $grantIds)->get();
            $count = 0;

            DB::transaction(function () use ($grants, $reason, &$count) {
                  foreach ($grants as $grant) {
                        $this->deleteGrant($grant, $reason);
                        $count++;
                  }
            });

            return $count;
      }
      /**
       * Notify grant user
       */
      protected function notifyUser(Grant $grant, string $status, ?string $note = null): void
      {
            $grant->user->notify(new GrantStatusNotification($grant, $status, $note));
      }
}
