<?php

namespace App\Policies;

use App\Models\Admin;
use Illuminate\Auth\Access\HandlesAuthorization;

class AdminPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the admin can view any admins.
     * All admins can view the admin list, but visibility of super_admins
     * is enforced in the Livewire component or in the `view` method.
     */
    public function viewAny(Admin $admin): bool
    {
        return true;
    }

    /**
     * Determine whether the admin can view a specific admin.
     * Non-super_admins cannot view super_admins.
     */
    public function view(Admin $admin, Admin $target): bool
    {
        if ($target->isSuperAdmin() && !$admin->isSuperAdmin()) {
            return false;
        }
        return true;
    }

    /**
     * Determine whether the admin can create a new admin.
     * Only super_admin can create another super_admin.
     * Any admin can create lower roles (admin or manager).
     *
     * @param Admin $admin Logged-in admin
     * @param string|null $role Role intended to create
     */
    public function create(Admin $admin, ?string $role = null): bool
    {
        if ($role === 'super_admin' && !$admin->isSuperAdmin()) {
            return false; // Only super_admin can create super_admin
        }
        return true; // All other roles allowed
    }

    /**
     * Determine whether the admin can update a specific admin.
     * - Non-super_admin cannot update a super_admin
     * - Admin can update own data but cannot promote themselves to super_admin
     */
    public function update(Admin $admin, Admin $target, ?string $newRole = null): bool
    {
        if ($target->isSuperAdmin() && !$admin->isSuperAdmin()) {
            return false;
        }

        // Prevent self-promotion to super_admin
        if ($admin->id === $target->id && $newRole === 'super_admin' && !$admin->isSuperAdmin()) {
            return false;
        }

        return true;
    }
    /**
     * Determine whether the admin can delete a specific admin.
     * Rules:
     * - Cannot delete self
     * - Only super_admin can delete super_admin
     * - All other admins can be deleted according to hierarchy
     */
    public function delete(Admin $admin, Admin $target): bool
    {
        if ($admin->id === $target->id) {
            return false; // Prevent deleting self
        }

        if ($target->isSuperAdmin() && !$admin->isSuperAdmin()) {
            return false; // Only super_admin can delete super_admin
        }

        return true;
    }

    /**
     * Determine whether the admin can assign a specific role.
     * Only super_admin can assign super_admin role.
     */
    public function assignRole($admin, string $role): bool
    {
        if (!method_exists($admin, 'isSuperAdmin')) {
            return false; // Safety check
        }

        if ($role === 'super_admin' && !$admin->isSuperAdmin()) {
            return false;
        }

        return true;
    }

    /**
     * Restore is not allowed for any admin.
     */
    public function restore(Admin $admin, Admin $target): bool
    {
        return false;
    }

    /**
     * Force delete is not allowed for any admin.
     */
    public function forceDelete(Admin $admin, Admin $target): bool
    {
        return false;
    }
}