<?php

namespace App\Models;

use App\Traits\VisibleToAdmin;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WalletReserve extends Model
{
    use HasFactory;
    use VisibleToAdmin;

    // This tells the trait how to reach the user who controls visibility
    protected function adminVisibilityRelation(): string
    {
        return 'account.profile.user';
    }

    /** ================= STATUSES ================= */
    public const STATUS_PENDING = 'pending';
    public const STATUS_APPROVED = 'approved';
    public const STATUS_REJECTED = 'rejected';

    /** ================= ACTION TYPES ================= */
    public const ACTION_CARD_TOPUP = 'card_topup';

    protected $fillable = [
        'account_id',
        'action_type',
        'action_id',
        'amount',
        'status',
        'locked_at',
        'approved_at',
    ];

    protected $casts = [
        'locked_at' => 'datetime',
        'approved_at' => 'datetime',
        'amount' => 'decimal:2',
    ];

    protected $attributes = [
        'status' => self::STATUS_PENDING,
    ];

    /** ================= RELATIONSHIPS ================= */

    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /** ================= STATUS HELPERS ================= */

    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isApproved(): bool
    {
        return $this->status === self::STATUS_APPROVED;
    }

    public function isRejected(): bool
    {
        return $this->status === self::STATUS_REJECTED;
    }
    public function scopePending($q)
    {
        return $q->where('status', self::STATUS_PENDING);
    }

}
